<?php
/**
 * Ars Medical Frankfurt - Notification Service
 * Handles email and SMS notifications to emergency contacts
 */

if (!defined('SECURE_ACCESS')) {
    define('SECURE_ACCESS', true);
}

require_once __DIR__ . '/db.php';

/**
 * Send notification to emergency contact when patient is admitted
 * 
 * @param int $patientId The patient ID
 * @param array $patientData Patient data array
 * @param string $notificationType 'email', 'sms', or 'both'
 * @return array Result with success status and message
 */
function sendEmergencyContactNotification($patientId, $patientData, $notificationType = 'both') {
    $results = [
        'success' => true,
        'email_sent' => false,
        'sms_sent' => false,
        'message' => ''
    ];
    
    // Validate patient data
    if (empty($patientData['emergency_contact']) || empty($patientData['emergency_phone'])) {
        return [
            'success' => false,
            'message' => 'Emergency contact information is missing'
        ];
    }
    
    $recipientName = $patientData['emergency_contact'];
    $recipientPhone = $patientData['emergency_phone'];
    $recipientEmail = !empty($patientData['emergency_email']) ? $patientData['emergency_email'] : null;
    
    $patientName = $patientData['first_name'] . ' ' . $patientData['last_name'];
    $admissionDate = date('d.m.Y', strtotime($patientData['admission_date']));
    $department = $patientData['department'];
    $roomNumber = $patientData['room_number'];
    $hospitalName = 'Ars Medical Frankfurt';
    $hospitalPhone = '+49 69 12345678';
    
    // Prepare message content
    $subject = "Patientenaufnahme: {$patientName} - {$hospitalName}";
    
    $message = "Sehr geehrte/r {$recipientName},\n\n";
    $message .= "wir informieren Sie, dass {$patientName} am {$admissionDate} in unserem Krankenhaus aufgenommen wurde.\n\n";
    $message .= "Aufnahmedetails:\n";
    $message .= "- Patient: {$patientName}\n";
    $message .= "- Abteilung: {$department}\n";
    $message .= "- Zimmer: {$roomNumber}\n";
    $message .= "- Aufnahmedatum: {$admissionDate}\n\n";
    $message .= "Bei Fragen erreichen Sie uns unter {$hospitalPhone}.\n\n";
    $message .= "Mit freundlichen Grüßen\n";
    $message .= "{$hospitalName}\n";
    $message .= "Windthorststraße 21, 65929 Frankfurt am Main";
    
    // Save notification record to database
    $db = getDB();
    $notificationId = $db->insert(
        "INSERT INTO notifications (patient_id, notification_type, recipient_name, recipient_email, recipient_phone, subject, message, status) VALUES (?, ?, ?, ?, ?, ?, ?, 'pending')",
        [$patientId, $notificationType, $recipientName, $recipientEmail, $recipientPhone, $subject, $message]
    );
    
    // Send Email
    if (($notificationType === 'email' || $notificationType === 'both') && !empty($recipientEmail)) {
        // Build HTML email content
        $htmlContent = buildPatientAdmissionEmailTemplate($patientData, $recipientName);
        $emailResult = sendEmail($recipientEmail, $recipientName, $subject, $htmlContent, true);
        $results['email_sent'] = $emailResult['success'];
        
        if (!$emailResult['success']) {
            $results['success'] = false;
            $results['message'] .= 'Email failed: ' . $emailResult['error'] . ' ';
            
            // Update notification status
            $db->update(
                "UPDATE notifications SET status = 'failed', error_message = ? WHERE id = ?",
                [$emailResult['error'], $notificationId]
            );
        }
    }
    
    // Send SMS
    if ($notificationType === 'sms' || $notificationType === 'both') {
        $smsMessage = "{$hospitalName}: {$patientName} wurde am {$admissionDate} aufgenommen. Abteilung: {$department}, Zimmer: {$roomNumber}. Tel: {$hospitalPhone}";
        $smsResult = sendSMS($recipientPhone, $smsMessage);
        $results['sms_sent'] = $smsResult['success'];
        
        if (!$smsResult['success']) {
            $results['success'] = false;
            $results['message'] .= 'SMS failed: ' . $smsResult['error'] . ' ';
            
            // Update notification status
            $db->update(
                "UPDATE notifications SET status = 'failed', error_message = CONCAT(IFNULL(error_message, ''), ?) WHERE id = ?",
                [' SMS: ' . $smsResult['error'], $notificationId]
            );
        }
    }
    
    // Update notification status to sent if successful
    if ($results['success'] || ($results['email_sent'] || $results['sms_sent'])) {
        $db->update(
            "UPDATE notifications SET status = 'sent', sent_at = NOW() WHERE id = ?",
            [$notificationId]
        );
    }
    
    if (empty($results['message'])) {
        $results['message'] = 'Notification sent successfully';
    }

    return $results;
}

/**
 * Build HTML email template for patient admission notification
 *
 * @param array $patientData Patient data array
 * @param string $recipientName Emergency contact name
 * @return string HTML email content
 */
function buildPatientAdmissionEmailTemplate($patientData, $recipientName) {
    $hospitalName = getSetting('hospital_name', 'Ars Medical Frankfurt');
    $hospitalPhone = getSetting('hospital_phone', '+49 69 12345678');
    $hospitalAddress = getSetting('hospital_address', 'Friedrichstraße 123, 60311 Frankfurt am Main');
    $hospitalEmail = getSetting('hospital_email', 'info@arsmedicafrankfurt.de');
    $logoUrl = getSetting('logo_url', '');

    $patientName = $patientData['first_name'] . ' ' . $patientData['last_name'];
    $admissionDate = date('d.m.Y', strtotime($patientData['admission_date']));
    $admissionTime = date('H:i', strtotime($patientData['created_at'] ?? 'now'));
    $department = $patientData['department'];
    $roomNumber = $patientData['room_number'];
    $doctor = $patientData['doctor'];
    $diagnosis = $patientData['diagnosis'];
    $bloodGroup = $patientData['blood_group'] ?? 'Nicht angegeben';
    $allergies = $patientData['allergies'] ?? 'Keine bekannt';

    $logoHtml = $logoUrl ? '<img src="' . $logoUrl . '" alt="' . $hospitalName . '" style="max-height: 60px; margin-bottom: 20px;">' : '<h1 style="color: #004B7C; margin-bottom: 20px;">' . $hospitalName . '</h1>';

    $html = <<<HTML
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Patientenaufnahme - {$hospitalName}</title>
    <style>
        body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; line-height: 1.6; color: #333; margin: 0; padding: 0; background-color: #f4f7f9; }
        .container { max-width: 600px; margin: 0 auto; background-color: #ffffff; }
        .header { background: linear-gradient(135deg, #004B7C 0%, #0066a8 100%); padding: 30px; text-align: center; color: white; }
        .header h1 { margin: 0; font-size: 24px; font-weight: 600; }
        .header p { margin: 10px 0 0 0; opacity: 0.9; }
        .content { padding: 30px; }
        .greeting { font-size: 18px; margin-bottom: 20px; color: #004B7C; }
        .info-box { background-color: #f8fafc; border-left: 4px solid #004B7C; padding: 20px; margin: 20px 0; border-radius: 0 8px 8px 0; }
        .info-box h3 { margin-top: 0; color: #004B7C; font-size: 16px; }
        .patient-details { background-color: #e8f4f8; padding: 20px; border-radius: 8px; margin: 20px 0; }
        .patient-details h3 { margin-top: 0; color: #004B7C; }
        .detail-row { display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid #e2e8f0; }
        .detail-row:last-child { border-bottom: none; }
        .detail-label { font-weight: 600; color: #64748b; }
        .detail-value { color: #1e293b; text-align: right; }
        .important { background-color: #fef3c7; border: 1px solid #f59e0b; padding: 15px; border-radius: 8px; margin: 20px 0; }
        .important h4 { margin-top: 0; color: #92400e; }
        .footer { background-color: #f1f5f9; padding: 20px; text-align: center; font-size: 14px; color: #64748b; }
        .footer a { color: #004B7C; text-decoration: none; }
        .contact-info { margin-top: 15px; padding-top: 15px; border-top: 1px solid #e2e8f0; }
        .btn { display: inline-block; background-color: #004B7C; color: white; padding: 12px 24px; text-decoration: none; border-radius: 6px; margin-top: 15px; }
        .security-notice { background-color: #d1fae5; border: 1px solid #10b981; padding: 15px; border-radius: 8px; margin: 20px 0; font-size: 14px; }
        .security-notice strong { color: #065f46; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            {$logoHtml}
            <h1>Patientenaufnahme-Benachrichtigung</h1>
            <p>{$hospitalName}</p>
        </div>

        <div class="content">
            <div class="greeting">
                Sehr geehrte/r <strong>{$recipientName}</strong>,
            </div>

            <p>wir informieren Sie, dass <strong>{$patientName}</strong> am <strong>{$admissionDate}</strong> um <strong>{$admissionTime} Uhr</strong> in unserem Krankenhaus aufgenommen wurde.</p>

            <div class="patient-details">
                <h3>Patienteninformationen</h3>
                <div class="detail-row">
                    <span class="detail-label">Patient:</span>
                    <span class="detail-value">{$patientName}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Aufnahmedatum:</span>
                    <span class="detail-value">{$admissionDate}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Aufnahmezeit:</span>
                    <span class="detail-value">{$admissionTime} Uhr</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Abteilung:</span>
                    <span class="detail-value">{$department}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Zimmernummer:</span>
                    <span class="detail-value">{$roomNumber}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Behandelnder Arzt:</span>
                    <span class="detail-value">{$doctor}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Diagnose:</span>
                    <span class="detail-value">{$diagnosis}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Blutgruppe:</span>
                    <span class="detail-value">{$bloodGroup}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Allergien:</span>
                    <span class="detail-value">{$allergies}</span>
                </div>
            </div>

            <div class="important">
                <h4>Wichtige Information</h4>
                <p>Die Patientin/der Patient befindet sich unter ärztlicher Betreuung. Bei Fragen zur Behandlung oder dem Gesundheitszustand wenden Sie sich bitte direkt an die Station oder den behandelnden Arzt.</p>
            </div>

            <div class="security-notice">
                <strong>Sicherheitshinweis:</strong> Diese E-Mail enthält vertrauliche medizinische Informationen. Wenn Sie diese Benachrichtigung irrtümlich erhalten haben, kontaktieren Sie uns bitte umgehend unter {$hospitalPhone}.
            </div>

            <div class="info-box">
                <h3>Kontakt & Besuchszeiten</h3>
                <p><strong>Telefon:</strong> {$hospitalPhone}<br>
                <strong>E-Mail:</strong> <a href="mailto:{$hospitalEmail}">{$hospitalEmail}</a><br>
                <strong>Adresse:</strong> {$hospitalAddress}</p>
                <p><strong>Allgemeine Besuchszeiten:</strong><br>
                Täglich: 14:00 - 19:00 Uhr</p>
            </div>
        </div>

        <div class="footer">
            <p><strong>{$hospitalName}</strong></p>
            <p>{$hospitalAddress}</p>
            <div class="contact-info">
                <p>Tel: {$hospitalPhone} | E-Mail: <a href="mailto:{$hospitalEmail}">{$hospitalEmail}</a></p>
            </div>
            <p style="margin-top: 20px; font-size: 12px; color: #94a3b8;">
                Diese E-Mail wurde automatisch generiert. Bitte antworten Sie nicht auf diese Nachricht.<br>
                &copy; " . date('Y') . " {$hospitalName}. Alle Rechte vorbehalten.
            </p>
        </div>
    </div>
</body>
</html>
HTML;

    return $html;
}

/**
 * Send email using PHP mail() or SMTP
 *
 * @param string $to Recipient email
 * @param string $name Recipient name
 * @param string $subject Email subject
 * @param string $body Email body
 * @param bool $isHtml Whether body is HTML (default: false)
 * @return array Result with success status
 */
function sendEmail($to, $name, $subject, $body, $isHtml = false) {
    // Get email configuration from site settings
    $fromEmail = getSetting('email_from_address', $config['from_email'] ?? 'noreply@arsmedicafrankfurt.de');
    $fromName = getSetting('email_from_name', $config['from_name'] ?? 'Ars Medical Frankfurt');
    $useSmtp = getSetting('email_use_smtp', '0') === '1';

    // Validate email
    if (!filter_var($to, FILTER_VALIDATE_EMAIL)) {
        return ['success' => false, 'error' => 'Invalid email address'];
    }

    // Build headers
    $headers = "From: {$fromName} <{$fromEmail}>\r\n";
    $headers .= "Reply-To: {$fromEmail}\r\n";
    $headers .= "X-Mailer: PHP/" . phpversion() . "\r\n";

    if ($isHtml) {
        $headers .= "MIME-Version: 1.0\r\n";
        $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
    }

    // For production, use SMTP instead of mail()
    if ($useSmtp && function_exists('stream_socket_client')) {
        return sendSMTPMail($to, $name, $subject, $body, $headers, $isHtml);
    }

    // Use PHP mail() function
    $success = mail($to, $subject, $body, $headers);

    return [
        'success' => $success,
        'error' => $success ? null : 'Failed to send email'
    ];
}

/**
 * Send email using SMTP (for production use)
 *
 * @param string $to Recipient email
 * @param string $name Recipient name
 * @param string $subject Email subject
 * @param string $body Email body
 * @param string $headers Email headers
 * @param bool $isHtml Whether body is HTML
 * @return array Result with success status
 */
function sendSMTPMail($to, $name, $subject, $body, $headers, $isHtml = false) {
    // Get SMTP settings from site_settings
    $smtpHost = getSetting('email_smtp_host', '');
    $smtpPort = getSetting('email_smtp_port', '587');
    $smtpUser = getSetting('email_smtp_user', '');
    $smtpPass = getSetting('email_smtp_pass', '');

    if (empty($smtpHost) || empty($smtpUser)) {
        // Fallback to mail()
        $fromEmail = getSetting('email_from_address', 'noreply@arsmedicafrankfurt.de');
        $fromName = getSetting('email_from_name', 'Ars Medical Frankfurt');
        $headers = "From: {$fromName} <{$fromEmail}>\r\n";
        if ($isHtml) {
            $headers .= "MIME-Version: 1.0\r\n";
            $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
        }
        $success = mail($to, $subject, $body, $headers);
        return ['success' => $success, 'error' => $success ? null : 'SMTP not configured, mail() failed'];
    }

    // SMTP connection would be implemented here
    // For production, consider using PHPMailer or SwiftMailer library
    // For now, fallback to mail() with proper headers
    $success = mail($to, $subject, $body, $headers);
    return ['success' => $success, 'error' => $success ? null : 'Failed to send email via SMTP'];
}

/**
 * Send SMS using SMS gateway API
 * 
 * @param string $phone Recipient phone number
 * @param string $message SMS message
 * @return array Result with success status
 */
function sendSMS($phone, $message) {
    // Get SMS configuration from site_settings
    $smsProvider = getSetting('sms_provider', '');
    $smsApiKey = getSetting('sms_api_key', '');

    // Validate phone number
    $phone = preg_replace('/[^\d+]/', '', $phone);
    if (strlen($phone) < 10) {
        return ['success' => false, 'error' => 'Invalid phone number'];
    }

    // Check if SMS provider is configured
    if (empty($smsProvider) || empty($smsApiKey)) {
        // Log that SMS would be sent (for demo/testing)
        error_log("SMS would be sent to {$phone}: {$message}");
        return ['success' => true, 'error' => null, 'note' => 'SMS provider not configured - logged only'];
    }

    // Send SMS via provider API
    switch ($smsProvider) {
        case 'twilio':
            return sendTwilioSMS($phone, $message);
        case 'vonage':
            return sendVonageSMS($phone, $message);
        case 'sms77':
            return sendSMS77($phone, $message);
        default:
            return ['success' => false, 'error' => 'Unknown SMS provider'];
    }
}

/**
 * Send SMS via Twilio
 */
function sendTwilioSMS($phone, $message) {
    $accountSid = getSetting('sms_account_sid', '');
    $authToken = getSetting('sms_auth_token', '');
    $fromNumber = getSetting('sms_from_number', '');
    
    if (empty($accountSid) || empty($authToken)) {
        return ['success' => false, 'error' => 'Twilio credentials not configured'];
    }
    
    $url = "https://api.twilio.com/2010-04-01/Accounts/{$accountSid}/Messages.json";
    
    $data = [
        'To' => $phone,
        'From' => $fromNumber,
        'Body' => $message
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_USERPWD, "{$accountSid}:{$authToken}");
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode === 201) {
        return ['success' => true, 'error' => null];
    } else {
        $error = json_decode($response, true);
        return ['success' => false, 'error' => $error['message'] ?? 'Twilio API error'];
    }
}

/**
 * Send SMS via Vonage (Nexmo)
 */
function sendVonageSMS($phone, $message) {
    $apiKey = getSetting('sms_api_key', '');
    $apiSecret = getSetting('sms_api_secret', '');
    $fromName = getSetting('sms_from_name', 'ArsMedical');
    
    if (empty($apiKey) || empty($apiSecret)) {
        return ['success' => false, 'error' => 'Vonage credentials not configured'];
    }
    
    $url = 'https://rest.nexmo.com/sms/json';
    
    $data = [
        'api_key' => $apiKey,
        'api_secret' => $apiSecret,
        'to' => $phone,
        'from' => $fromName,
        'text' => $message
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    
    $response = curl_exec($ch);
    curl_close($ch);
    
    $result = json_decode($response, true);
    
    if (isset($result['messages'][0]['status']) && $result['messages'][0]['status'] == '0') {
        return ['success' => true, 'error' => null];
    } else {
        return ['success' => false, 'error' => $result['messages'][0]['error-text'] ?? 'Vonage API error'];
    }
}

/**
 * Send SMS via SMS77 (German provider)
 */
function sendSMS77($phone, $message) {
    $apiKey = getSetting('sms_api_key', '');
    
    if (empty($apiKey)) {
        return ['success' => false, 'error' => 'SMS77 API key not configured'];
    }
    
    $url = 'https://gateway.sms77.io/api/sms';
    
    $data = [
        'to' => $phone,
        'text' => $message,
        'from' => $config['sms_from_name'] ?? 'ArsMedical'
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'X-Api-Key: ' . $apiKey
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode === 200) {
        return ['success' => true, 'error' => null];
    } else {
        return ['success' => false, 'error' => 'SMS77 API error: ' . $response];
    }
}

/**
 * Get notification configuration
 * 
 * @return array Configuration array
 */
function getNotificationConfig() {
    // Default configuration - update these values for production
    return [
        // Email settings
        'from_email' => 'noreply@arsmedicafrankfurt.de',
        'from_name' => 'Ars Medical Frankfurt',
        'use_smtp' => false,
        'smtp_host' => '',
        'smtp_port' => 587,
        'smtp_user' => '',
        'smtp_pass' => '',
        
        // SMS settings
        'sms_provider' => '', // 'twilio', 'vonage', 'sms77', or '' for disabled
        'sms_api_key' => '',
        'sms_api_secret' => '',
        'sms_account_sid' => '', // For Twilio
        'sms_auth_token' => '', // For Twilio
        'sms_from_number' => '', // For Twilio
        'sms_from_name' => 'ArsMedical' // For Vonage/SMS77
    ];
}

/**
 * Get notification history for a patient
 * 
 * @param int $patientId Patient ID
 * @return array Array of notifications
 */
function getPatientNotifications($patientId) {
    try {
        $db = getDB();
        return $db->fetchAll(
            "SELECT * FROM notifications WHERE patient_id = ? ORDER BY created_at DESC",
            [$patientId]
        );
    } catch (Exception $e) {
        return [];
    }
}

/**
 * Resend failed notification
 * 
 * @param int $notificationId Notification ID
 * @return array Result
 */
function resendNotification($notificationId) {
    try {
        $db = getDB();
        $notification = $db->fetchOne(
            "SELECT * FROM notifications WHERE id = ?",
            [$notificationId]
        );
        
        if (!$notification) {
            return ['success' => false, 'error' => 'Notification not found'];
        }
        
        // Resend based on type
        $results = ['success' => true];
        
        if ($notification['notification_type'] === 'email' || $notification['notification_type'] === 'both') {
            if (!empty($notification['recipient_email'])) {
                $emailResult = sendEmail(
                    $notification['recipient_email'],
                    $notification['recipient_name'],
                    $notification['subject'],
                    $notification['message']
                );
                $results['email_sent'] = $emailResult['success'];
            }
        }
        
        if ($notification['notification_type'] === 'sms' || $notification['notification_type'] === 'both') {
            if (!empty($notification['recipient_phone'])) {
                // Extract short message from full message
                $lines = explode("\n", $notification['message']);
                $smsMessage = $lines[0] . ' ' . $lines[2] ?? $notification['message'];
                
                $smsResult = sendSMS($notification['recipient_phone'], $smsMessage);
                $results['sms_sent'] = $smsResult['success'];
            }
        }
        
        // Update status
        $status = ($results['email_sent'] ?? false) || ($results['sms_sent'] ?? false) ? 'sent' : 'failed';
        $db->update(
            "UPDATE notifications SET status = ?, sent_at = NOW() WHERE id = ?",
            [$status, $notificationId]
        );
        
        return $results;
        
    } catch (Exception $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}
