<?php
/**
 * Ars Medical Frankfurt - Utility Functions
 */

if (!defined('SECURE_ACCESS')) {
    define('SECURE_ACCESS', true);
}

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once __DIR__ . '/db.php';

// Ensure CSRF constant is defined (defined in config.php, but added here as fallback)
if (!defined('CSRF_TOKEN_NAME')) {
    define('CSRF_TOKEN_NAME', 'csrf_token');
}

// Ensure CSRF functions are available (defined in config.php, but added here as fallback)
if (!function_exists('generateCSRFToken')) {
    /**
     * Generate CSRF Token
     */
    function generateCSRFToken() {
        if (empty($_SESSION[CSRF_TOKEN_NAME])) {
            $_SESSION[CSRF_TOKEN_NAME] = bin2hex(random_bytes(32));
        }
        return $_SESSION[CSRF_TOKEN_NAME];
    }
}

if (!function_exists('validateCSRFToken')) {
    /**
     * Validate CSRF Token
     */
    function validateCSRFToken($token) {
        // Debug logging for troubleshooting
        if (empty($token)) {
            error_log('CSRF validation failed: empty token provided');
            return false;
        }
        if (empty($_SESSION[CSRF_TOKEN_NAME])) {
            error_log('CSRF validation failed: no token in session');
            return false;
        }
        return hash_equals($_SESSION[CSRF_TOKEN_NAME], $token);
    }
}

/**
 * Check if user is logged in
 */
function isLoggedIn() {
    return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
}

/**
 * Check if user is admin
 */
function isAdmin() {
    return isLoggedIn() && isset($_SESSION['user_role']) && $_SESSION['user_role'] === 'admin';
}

/**
 * Require authentication
 */
function requireAuth() {
    if (!isLoggedIn()) {
        header('Location: /admin/login.php');
        exit;
    }
}

/**
 * Require admin privileges
 */
function requireAdmin() {
    requireAuth();
    if (!isAdmin()) {
        header('HTTP/1.1 403 Forbidden');
        exit('Access denied');
    }
}

/**
 * Get current user ID
 */
function getCurrentUserId() {
    return $_SESSION['user_id'] ?? null;
}

/**
 * Get current user data
 */
function getCurrentUser() {
    if (!isLoggedIn()) {
        return null;
    }
    
    try {
        $db = getDB();
        return $db->fetchOne("SELECT id, username, email, full_name, role FROM users WHERE id = ?", [getCurrentUserId()]);
    } catch (Exception $e) {
        return null;
    }
}

/**
 * Format currency (Euro)
 */
function formatCurrency($amount) {
    return number_format($amount, 2, ',', '.') . ' €';
}

/**
 * Format date (German)
 */
function formatDate($date, $format = 'd.m.Y') {
    if (empty($date)) return '-';
    return date($format, strtotime($date));
}

/**
 * Format datetime (German)
 */
function formatDateTime($datetime) {
    if (empty($datetime)) return '-';
    return date('d.m.Y H:i', strtotime($datetime));
}

/**
 * Generate invoice number
 */
function generateInvoiceNumber() {
    $prefix = 'RE-' . date('Y');
    $db = getDB();
    
    // Get the last invoice number for this year
    $result = $db->fetchOne(
        "SELECT invoice_number FROM invoices WHERE invoice_number LIKE ? ORDER BY id DESC LIMIT 1",
        [$prefix . '%']
    );
    
    if ($result) {
        // Extract number and increment
        preg_match('/RE-\d{4}-(\d+)/', $result['invoice_number'], $matches);
        $nextNum = isset($matches[1]) ? intval($matches[1]) + 1 : 1;
    } else {
        $nextNum = 1;
    }
    
    return $prefix . '-' . str_pad($nextNum, 4, '0', STR_PAD_LEFT);
}

/**
 * Calculate age from birth date
 */
function calculateAge($birthDate) {
    $birth = new DateTime($birthDate);
    $today = new DateTime();
    $diff = $today->diff($birth);
    return $diff->y;
}

/**
 * Validate email address
 */
function isValidEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Validate German postal code
 */
function isValidPostalCode($plz) {
    return preg_match('/^\d{5}$/', $plz);
}

/**
 * Validate phone number (basic)
 */
function isValidPhone($phone) {
    // Allow +, numbers, spaces, dashes, parentheses
    return preg_match('/^[\+\d\s\-\(\)]{6,20}$/', $phone);
}

/**
 * Send JSON response
 */
function jsonResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data);
    exit;
}

/**
 * Send error response
 */
function errorResponse($message, $statusCode = 400) {
    jsonResponse(['success' => false, 'error' => $message], $statusCode);
}

/**
 * Send success response
 */
function successResponse($data = null, $message = 'Success') {
    $response = ['success' => true, 'message' => $message];
    if ($data !== null) {
        $response['data'] = $data;
    }
    jsonResponse($response);
}

/**
 * Get patient by ID
 */
function getPatient($id) {
    try {
        $db = getDB();
        return $db->fetchOne("SELECT * FROM patients WHERE id = ?", [$id]);
    } catch (Exception $e) {
        return null;
    }
}

/**
 * Get all patients with optional filters
 */
function getPatients($filters = []) {
    try {
        $db = getDB();
        $sql = "SELECT * FROM patients WHERE 1=1";
        $params = [];
        
        if (!empty($filters['status'])) {
            $sql .= " AND status = ?";
            $params[] = $filters['status'];
        }
        
        if (!empty($filters['department'])) {
            $sql .= " AND department = ?";
            $params[] = $filters['department'];
        }
        
        if (!empty($filters['search'])) {
            $sql .= " AND (first_name LIKE ? OR last_name LIKE ? OR diagnosis LIKE ?)";
            $searchTerm = '%' . $filters['search'] . '%';
            $params[] = $searchTerm;
            $params[] = $searchTerm;
            $params[] = $searchTerm;
        }
        
        $sql .= " ORDER BY created_at DESC";
        
        return $db->fetchAll($sql, $params);
    } catch (Exception $e) {
        return [];
    }
}

/**
 * Get invoice by ID
 */
function getInvoice($id) {
    try {
        $db = getDB();
        return $db->fetchOne("SELECT * FROM invoices WHERE id = ?", [$id]);
    } catch (Exception $e) {
        return null;
    }
}

/**
 * Get all invoices with optional filters
 */
function getInvoices($filters = []) {
    try {
        $db = getDB();
        $sql = "SELECT * FROM invoices WHERE 1=1";
        $params = [];
        
        if (!empty($filters['status'])) {
            $sql .= " AND status = ?";
            $params[] = $filters['status'];
        }
        
        if (!empty($filters['patient_id'])) {
            $sql .= " AND patient_id = ?";
            $params[] = $filters['patient_id'];
        }
        
        if (!empty($filters['search'])) {
            $sql .= " AND (invoice_number LIKE ? OR patient_name LIKE ?)";
            $searchTerm = '%' . $filters['search'] . '%';
            $params[] = $searchTerm;
            $params[] = $searchTerm;
        }
        
        $sql .= " ORDER BY created_at DESC";
        
        return $db->fetchAll($sql, $params);
    } catch (Exception $e) {
        return [];
    }
}

/**
 * Get dashboard statistics (for sidebar/public use)
 * Note: Admin dashboard uses api/dashboard.php
 */
function getDashboardStatsSummary() {
    try {
        $db = getDB();
        $stats = [];

        // Check if tables exist
        $patientsExists = $db->fetchOne("SHOW TABLES LIKE 'patients'");
        $invoicesExists = $db->fetchOne("SHOW TABLES LIKE 'invoices'");

        // Total admitted patients
        if ($patientsExists) {
            $result = $db->fetchOne("SELECT COUNT(*) as count FROM patients WHERE status = 'aufgenommen'");
            $stats['total_patients'] = $result['count'];

            // Today's admissions
            $result = $db->fetchOne("SELECT COUNT(*) as count FROM patients WHERE DATE(admission_date) = CURDATE()");
            $stats['today_admissions'] = $result['count'];

            // Recent patients
            $stats['recent_patients'] = $db->fetchAll(
                "SELECT id, first_name, last_name, room_number, doctor, admission_date, status FROM patients ORDER BY created_at DESC LIMIT 5"
            );
        } else {
            $stats['total_patients'] = 0;
            $stats['today_admissions'] = 0;
            $stats['recent_patients'] = [];
        }

        // Open invoices
        if ($invoicesExists) {
            $result = $db->fetchOne("SELECT COUNT(*) as count FROM invoices WHERE status = 'offen'");
            $stats['open_invoices'] = $result['count'];

            // Monthly revenue
            $result = $db->fetchOne("SELECT COALESCE(SUM(total_amount), 0) as total FROM invoices WHERE MONTH(invoice_date) = MONTH(CURDATE()) AND YEAR(invoice_date) = YEAR(CURDATE())");
            $stats['monthly_revenue'] = $result['total'];
        } else {
            $stats['open_invoices'] = 0;
            $stats['monthly_revenue'] = 0;
        }

        return $stats;
    } catch (Exception $e) {
        return [
            'total_patients' => 0,
            'today_admissions' => 0,
            'open_invoices' => 0,
            'monthly_revenue' => 0,
            'recent_patients' => []
        ];
    }
}

/**
 * Get departments list
 */
function getDepartments() {
    return [
        'Kardiologie' => 'Kardiologie',
        'Orthopädie' => 'Orthopädie',
        'Gynäkologie' => 'Gynäkologie',
        'Radiologie' => 'Radiologie',
        'Onkologie' => 'Onkologie',
        'Neurologie' => 'Neurologie',
        'Innere Medizin' => 'Innere Medizin'
    ];
}

/**
 * Get insurance companies
 */
function getInsuranceCompanies() {
    return [
        'AOK' => 'AOK',
        'TK' => 'Techniker Krankenkasse',
        'DAK' => 'DAK-Gesundheit',
        'Barmer' => 'Barmer',
        'KKH' => 'KKH',
        'hkk' => 'hkk',
        'IKK' => 'IKK classic',
        'Privat' => 'Privatversicherung',
        'Selbstzahler' => 'Selbstzahler'
    ];
}

/**
 * Get blood groups
 */
function getBloodGroups() {
    return [
        '' => 'Unbekannt',
        'A+' => 'A+',
        'A-' => 'A-',
        'B+' => 'B+',
        'B-' => 'B-',
        'AB+' => 'AB+',
        'AB-' => 'AB-',
        '0+' => '0+',
        '0-' => '0-'
    ];
}

/**
 * Get doctors list
 */
function getDoctors() {
    try {
        $db = getDB();
        return $db->fetchAll("SELECT id, full_name, email FROM users WHERE role = 'doctor' AND is_active = 1 ORDER BY full_name");
    } catch (Exception $e) {
        return [
            ['id' => 1, 'full_name' => 'Dr. Klaus Schmidt'],
            ['id' => 2, 'full_name' => 'Dr. Maria Weber'],
            ['id' => 3, 'full_name' => 'Dr. Thomas Müller'],
            ['id' => 4, 'full_name' => 'Dr. Sarah Hoffmann']
        ];
    }
}

/**
 * Get site setting value by key
 * @param string $key Setting key
 * @param string $default Default value if setting not found
 * @return string Setting value or default
 */
function getSetting($key, $default = '') {
    static $settings = null;
    
    // Load settings once per request
    if ($settings === null) {
        try {
            $db = getDB();
            $results = $db->fetchAll("SELECT setting_key, setting_value FROM site_settings WHERE is_public = 1");
            $settings = [];
            foreach ($results as $row) {
                $settings[$row['setting_key']] = $row['setting_value'];
            }
        } catch (Exception $e) {
            // Return defaults if database error
            $settings = [];
        }
    }
    
    return isset($settings[$key]) ? $settings[$key] : $default;
}

/**
 * Get all site settings
 * @param bool $includePrivate Include private settings (admin only)
 * @return array All settings
 */
function getAllSettings($includePrivate = false) {
    try {
        $db = getDB();
        $sql = "SELECT * FROM site_settings WHERE 1=1";
        $params = [];
        
        if (!$includePrivate) {
            $sql .= " AND is_public = 1";
        }
        
        $sql .= " ORDER BY setting_key ASC";
        
        return $db->fetchAll($sql, $params);
    } catch (Exception $e) {
        return [];
    }
}

/**
 * Display flash message
 */
function showFlashMessage() {
    if (isset($_SESSION['flash_message'])) {
        $message = $_SESSION['flash_message'];
        $type = $_SESSION['flash_type'] ?? 'info';
        unset($_SESSION['flash_message']);
        unset($_SESSION['flash_type']);
        
        $colors = [
            'success' => 'bg-green-100 border-green-400 text-green-700',
            'error' => 'bg-red-100 border-red-400 text-red-700',
            'warning' => 'bg-yellow-100 border-yellow-400 text-yellow-700',
            'info' => 'bg-blue-100 border-blue-400 text-blue-700'
        ];
        
        $color = $colors[$type] ?? $colors['info'];
        
        echo '<div class="' . $color . ' px-4 py-3 rounded mb-4 border" role="alert">';
        echo '<span class="block sm:inline">' . e($message) . '</span>';
        echo '</div>';
    }
}

/**
 * Set flash message
 */
function setFlashMessage($message, $type = 'info') {
    $_SESSION['flash_message'] = $message;
    $_SESSION['flash_type'] = $type;
}
