-- Ars Medical Frankfurt Database Schema
-- MySQL/MariaDB Database Setup
-- Run this script to create the database and tables

-- Users table for admin authentication
CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL UNIQUE,
    password_hash VARCHAR(255) NOT NULL,
    email VARCHAR(100) NOT NULL,
    full_name VARCHAR(100) NOT NULL,
    role ENUM('admin', 'doctor', 'staff') DEFAULT 'staff',
    is_active BOOLEAN DEFAULT TRUE,
    last_login DATETIME NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_username (username),
    INDEX idx_email (email)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Patients table
CREATE TABLE IF NOT EXISTS patients (
    id INT AUTO_INCREMENT PRIMARY KEY,
    first_name VARCHAR(100) NOT NULL,
    last_name VARCHAR(100) NOT NULL,
    birth_date DATE NOT NULL,
    age INT NOT NULL,
    street VARCHAR(255) NOT NULL,
    postal_code VARCHAR(10) NOT NULL,
    city VARCHAR(100) NOT NULL,
    phone VARCHAR(20) NOT NULL,
    email VARCHAR(100) NULL,
    emergency_contact VARCHAR(100) NOT NULL,
    emergency_phone VARCHAR(20) NOT NULL,
    diagnosis TEXT NOT NULL,
    department VARCHAR(50) NOT NULL,
    doctor VARCHAR(100) NOT NULL,
    room_number VARCHAR(20) NOT NULL,
    blood_group VARCHAR(5) NULL,
    admission_date DATE NOT NULL,
    allergies TEXT NULL,
    conditions TEXT NULL,
    insurance VARCHAR(50) NOT NULL,
    insurance_number VARCHAR(50) NOT NULL,
    insurance_type ENUM('gesetzlich', 'privat', 'beihilfe') DEFAULT 'gesetzlich',
    status ENUM('aufgenommen', 'entlassen') DEFAULT 'aufgenommen',
    discharge_date DATE NULL,
    created_by INT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_name (last_name, first_name),
    INDEX idx_status (status),
    INDEX idx_department (department),
    INDEX idx_admission_date (admission_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Invoices table
CREATE TABLE IF NOT EXISTS invoices (
    id INT AUTO_INCREMENT PRIMARY KEY,
    invoice_number VARCHAR(50) NOT NULL UNIQUE,
    patient_id INT NOT NULL,
    patient_name VARCHAR(200) NOT NULL,
    invoice_date DATE NOT NULL,
    due_date DATE NOT NULL,
    treatment_cost DECIMAL(10, 2) DEFAULT 0.00,
    medication_cost DECIMAL(10, 2) DEFAULT 0.00,
    room_cost_per_day DECIMAL(10, 2) DEFAULT 0.00,
    stay_days INT DEFAULT 1,
    doctor_fee DECIMAL(10, 2) DEFAULT 0.00,
    other_costs DECIMAL(10, 2) DEFAULT 0.00,
    subtotal DECIMAL(10, 2) NOT NULL,
    tax_amount DECIMAL(10, 2) NOT NULL,
    tax_rate DECIMAL(5, 2) DEFAULT 19.00,
    total_amount DECIMAL(10, 2) NOT NULL,
    status ENUM('offen', 'bezahlt', 'ueberfaellig', 'storniert') DEFAULT 'offen',
    paid_date DATE NULL,
    notes TEXT NULL,
    created_by INT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (patient_id) REFERENCES patients(id) ON DELETE CASCADE,
    FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_invoice_number (invoice_number),
    INDEX idx_status (status),
    INDEX idx_invoice_date (invoice_date),
    INDEX idx_patient_id (patient_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Login attempts tracking (for rate limiting)
CREATE TABLE IF NOT EXISTS login_attempts (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL,
    ip_address VARCHAR(45) NOT NULL,
    attempted_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    success BOOLEAN DEFAULT FALSE,
    INDEX idx_username (username),
    INDEX idx_ip_address (ip_address),
    INDEX idx_attempted_at (attempted_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Activity log for audit trail
CREATE TABLE IF NOT EXISTS activity_log (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NULL,
    action VARCHAR(50) NOT NULL,
    entity_type VARCHAR(50) NULL,
    entity_id INT NULL,
    details TEXT NULL,
    ip_address VARCHAR(45) NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_user_id (user_id),
    INDEX idx_action (action),
    INDEX idx_created_at (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Site Settings table for global configuration
CREATE TABLE IF NOT EXISTS site_settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    setting_key VARCHAR(100) NOT NULL UNIQUE,
    setting_value TEXT NULL,
    setting_type ENUM('text', 'textarea', 'email', 'phone', 'url', 'number', 'boolean') DEFAULT 'text',
    description VARCHAR(255) NULL,
    is_public BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_setting_key (setting_key),
    INDEX idx_is_public (is_public)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert default site settings
INSERT INTO site_settings (setting_key, setting_value, setting_type, description, is_public) VALUES
('hospital_name', 'Ars Medical Frankfurt', 'text', 'Hospital name displayed across the site', TRUE),
('hospital_phone', '+49 69 1234 5678', 'phone', 'Main hospital phone number', TRUE),
('hospital_email', 'info@arsmedicafrankfurt.de', 'email', 'Main hospital email address', TRUE),
('hospital_address', 'Friedrichstraße 123, 60311 Frankfurt am Main', 'textarea', 'Hospital full address', TRUE),
('emergency_phone', '112', 'phone', 'Emergency phone number', TRUE),
('ambulance_phone', '+49 69 1234 5679', 'phone', 'Ambulance/emergency department phone', TRUE),
('appointment_phone', '+49 69 1234 5680', 'phone', 'Appointment booking phone number', TRUE),
('fax_number', '+49 69 1234 5699', 'phone', 'Hospital fax number', TRUE),
('opening_hours', 'Mo-Fr: 08:00 - 18:00 Uhr', 'text', 'Regular opening hours', TRUE),
('emergency_hours', '24/7', 'text', 'Emergency department hours', TRUE),
('facebook_url', '', 'url', 'Facebook page URL', TRUE),
('twitter_url', '', 'url', 'Twitter/X page URL', TRUE),
('linkedin_url', '', 'url', 'LinkedIn page URL', TRUE),
('instagram_url', '', 'url', 'Instagram page URL', TRUE),
('meta_title', 'Ars Medical Frankfurt - Ihr Krankenhaus mit Tradition', 'text', 'Default SEO meta title', TRUE),
('meta_description', 'Ars Medical Frankfurt - Modernste Medizin mit 168 Jahren Tradition. Kardiologie, Orthopädie, Gynäkologie & mehr.', 'textarea', 'Default SEO meta description', TRUE),
('analytics_code', '', 'textarea', 'Google Analytics or other tracking code (private)', FALSE),
('maintenance_mode', '0', 'boolean', 'Enable maintenance mode (0=off, 1=on)', FALSE),
('logo_url', '', 'url', 'Hospital logo URL (recommended: 200x60px PNG/SVG)', TRUE),
('logo_favicon', '', 'url', 'Favicon URL (32x32px or 64x64px)', TRUE),

-- Email Configuration
('email_from_address', 'noreply@arsmedicafrankfurt.de', 'email', 'Default from email address', FALSE),
('email_from_name', 'Ars Medical Frankfurt', 'text', 'Default from name', FALSE),
('email_use_smtp', '0', 'boolean', 'Use SMTP for emails (1=yes, 0=no)', FALSE),
('email_smtp_host', '', 'text', 'SMTP server hostname', FALSE),
('email_smtp_port', '587', 'number', 'SMTP server port', FALSE),
('email_smtp_user', '', 'text', 'SMTP username', FALSE),
('email_smtp_pass', '', 'text', 'SMTP password', FALSE),

-- SMS Configuration
('sms_provider', '', 'text', 'SMS provider (twilio, vonage, sms77, or empty)', FALSE),
('sms_api_key', '', 'text', 'SMS API Key', FALSE),
('sms_api_secret', '', 'text', 'SMS API Secret', FALSE),
('sms_account_sid', '', 'text', 'SMS Account SID (Twilio)', FALSE),
('sms_auth_token', '', 'text', 'SMS Auth Token (Twilio)', FALSE),
('sms_from_number', '', 'text', 'SMS From Number', FALSE),
('sms_from_name', 'ArsMedical', 'text', 'SMS From Name (Vonage/SMS77)', FALSE);

-- Notifications table for tracking sent notifications
CREATE TABLE IF NOT EXISTS notifications (
    id INT AUTO_INCREMENT PRIMARY KEY,
    patient_id INT NOT NULL,
    notification_type ENUM('email', 'sms', 'both') NOT NULL,
    recipient_name VARCHAR(100) NOT NULL,
    recipient_email VARCHAR(100) NULL,
    recipient_phone VARCHAR(20) NULL,
    subject VARCHAR(255) NULL,
    message TEXT NOT NULL,
    status ENUM('pending', 'sent', 'failed') DEFAULT 'pending',
    sent_at TIMESTAMP NULL,
    error_message TEXT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (patient_id) REFERENCES patients(id) ON DELETE CASCADE,
    INDEX idx_patient_id (patient_id),
    INDEX idx_status (status),
    INDEX idx_created_at (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert default admin user
-- Password: admin123 (change this in production!)
-- To generate a new hash: echo password_hash('your_password', PASSWORD_BCRYPT);
INSERT INTO users (username, password_hash, email, full_name, role, is_active) VALUES
('admin', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'admin@arsmedicafrankfurt.de', 'Administrator', 'admin', TRUE);

-- Insert sample doctors
INSERT INTO users (username, password_hash, email, full_name, role, is_active) VALUES
('dr.schmidt', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'k.schmidt@arsmedicafrankfurt.de', 'Dr. Klaus Schmidt', 'doctor', TRUE),
('dr.weber', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'm.weber@arsmedicafrankfurt.de', 'Dr. Maria Weber', 'doctor', TRUE),
('dr.mueller', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 't.mueller@arsmedicafrankfurt.de', 'Dr. Thomas Müller', 'doctor', TRUE),
('dr.hoffmann', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 's.hoffmann@arsmedicafrankfurt.de', 'Dr. Sarah Hoffmann', 'doctor', TRUE);

-- Insert sample patients
INSERT INTO patients (first_name, last_name, birth_date, age, street, postal_code, city, phone, email, emergency_contact, emergency_phone, diagnosis, department, doctor, room_number, blood_group, admission_date, allergies, conditions, insurance, insurance_number, insurance_type, status, created_at) VALUES
('Max', 'Mustermann', '1980-05-15', 44, 'Musterstraße 12', '60311', 'Frankfurt am Main', '+49 69 1234567', 'max.mustermann@email.de', 'Erika Mustermann', '+49 69 7654321', 'Koronare Herzkrankheit', 'Kardiologie', 'Dr. Klaus Schmidt', '201-A', 'A+', '2026-02-20', 'Keine', 'Bluthochdruck', 'AOK', 'A123456789', 'gesetzlich', 'aufgenommen', NOW()),
('Anna', 'Schmidt', '1975-08-22', 50, 'Hauptstraße 45', '65929', 'Frankfurt am Main', '+49 69 9876543', 'anna.schmidt@email.de', 'Peter Schmidt', '+49 69 3456789', 'Gebärmuttermyom', 'Gynäkologie', 'Dr. Maria Weber', '305-B', '0+', '2026-02-22', 'Penicillin', 'Keine', 'TK', 'T987654321', 'gesetzlich', 'aufgenommen', NOW()),
('Hans', 'Müller', '1960-03-10', 65, 'Berliner Straße 78', '60313', 'Frankfurt am Main', '+49 69 4567890', NULL, 'Maria Müller', '+49 69 0987654', 'Hüftarthrose', 'Orthopädie', 'Dr. Thomas Müller', '412-C', 'B+', '2026-02-18', 'Keine', 'Diabetes Typ 2', 'DAK', 'D456789012', 'gesetzlich', 'aufgenommen', NOW()),
('Lisa', 'Weber', '1990-11-28', 35, 'Kaiserstraße 23', '60311', 'Frankfurt am Main', '+49 69 2345678', 'lisa.weber@email.de', 'Thomas Weber', '+49 69 8765432', 'Mammakarzinom', 'Onkologie', 'Dr. Sarah Hoffmann', '508-D', 'AB-', '2026-02-15', 'Jod', 'Asthma', 'Barmer', 'B234567890', 'gesetzlich', 'aufgenommen', NOW());

-- Insert sample invoices
INSERT INTO invoices (invoice_number, patient_id, patient_name, invoice_date, due_date, treatment_cost, medication_cost, room_cost_per_day, stay_days, doctor_fee, other_costs, subtotal, tax_amount, tax_rate, total_amount, status, created_at) VALUES
('RE-2026-0001', 1, 'Max Mustermann', '2026-02-25', '2026-03-11', 2500.00, 450.00, 150.00, 5, 800.00, 200.00, 4650.00, 883.50, 19.00, 5533.50, 'offen', NOW()),
('RE-2026-0002', 2, 'Anna Schmidt', '2026-02-24', '2026-03-10', 3200.00, 280.00, 150.00, 3, 600.00, 150.00, 4880.00, 927.20, 19.00, 5807.20, 'bezahlt', NOW()),
('RE-2026-0003', 3, 'Hans Müller', '2026-02-23', '2026-03-09', 8500.00, 1200.00, 150.00, 7, 1500.00, 500.00, 12750.00, 2422.50, 19.00, 15172.50, 'offen', NOW());

-- Verify tables created
SHOW TABLES;

-- Display counts
SELECT 'Users' as table_name, COUNT(*) as count FROM users
UNION ALL
SELECT 'Patients', COUNT(*) FROM patients
UNION ALL
SELECT 'Invoices', COUNT(*) FROM invoices;