<?php
/**
 * Ars Medical Frankfurt - Logo Upload API
 * Handles logo and favicon file uploads
 */

if (!defined('SECURE_ACCESS')) {
    define('SECURE_ACCESS', true);
}

require_once __DIR__ . '/../includes/functions.php';

// Require admin authentication
requireAdmin();

header('Content-Type: application/json');

$method = $_SERVER['REQUEST_METHOD'];

if ($method !== 'POST') {
    errorResponse('Method not allowed', 405);
}

$uploadType = $_POST['type'] ?? 'logo'; // 'logo', 'favicon', or 'footer_image'

// Validate upload type
if (!in_array($uploadType, ['logo', 'favicon', 'footer_image'])) {
    errorResponse('Invalid upload type');
}

// Check if file was uploaded
if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
    $error = $_FILES['file']['error'] ?? 'No file uploaded';
    $errorMessages = [
        UPLOAD_ERR_INI_SIZE => 'File exceeds server maximum size',
        UPLOAD_ERR_FORM_SIZE => 'File exceeds form maximum size',
        UPLOAD_ERR_PARTIAL => 'File was only partially uploaded',
        UPLOAD_ERR_NO_FILE => 'No file was uploaded',
        UPLOAD_ERR_NO_TMP_DIR => 'Missing temporary folder',
        UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk',
        UPLOAD_ERR_EXTENSION => 'A PHP extension stopped the upload'
    ];
    errorResponse($errorMessages[$error] ?? 'Upload failed');
}

$file = $_FILES['file'];

// Validate file type
$allowedTypes = ['image/png', 'image/jpeg', 'image/svg+xml', 'image/webp'];
$finfo = finfo_open(FILEINFO_MIME_TYPE);
$mimeType = finfo_file($finfo, $file['tmp_name']);
finfo_close($finfo);

if (!in_array($mimeType, $allowedTypes)) {
    errorResponse('Invalid file type. Allowed: PNG, JPEG, SVG, WebP');
}

// Validate file size (max 2MB)
$maxSize = 2 * 1024 * 1024; // 2MB
if ($file['size'] > $maxSize) {
    errorResponse('File too large. Maximum size: 2MB');
}

// Create uploads directory if it doesn't exist
$uploadDir = __DIR__ . '/../uploads/logos/';
if (!is_dir($uploadDir)) {
    mkdir($uploadDir, 0755, true);
}

// Generate safe filename
$extension = pathinfo($file['name'], PATHINFO_EXTENSION);
$filename = 'logo_' . time() . '_' . bin2hex(random_bytes(8)) . '.' . $extension;
$filepath = $uploadDir . $filename;

// Move uploaded file
if (!move_uploaded_file($file['tmp_name'], $filepath)) {
    errorResponse('Failed to save file');
}

// Generate URL path
$protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
$host = $_SERVER['HTTP_HOST'];
$fileUrl = $protocol . '://' . $host . '/uploads/logos/' . $filename;

// Update database with new logo URL
$settingKeys = [
    'logo' => 'logo_url',
    'favicon' => 'logo_favicon',
    'footer_image' => 'footer_image_url'
];
$settingDescriptions = [
    'logo' => 'Hospital logo URL',
    'favicon' => 'Favicon URL',
    'footer_image' => 'Footer image URL'
];
$settingKey = $settingKeys[$uploadType] ?? 'logo_url';

try {
    $db = getDB();

    // Check if setting exists
    $existing = $db->fetchOne(
        "SELECT id FROM site_settings WHERE setting_key = ?",
        [$settingKey]
    );

    if ($existing) {
        // Get old logo URL to delete later
        $oldSetting = $db->fetchOne(
            "SELECT setting_value FROM site_settings WHERE setting_key = ?",
            [$settingKey]
        );
        $oldUrl = $oldSetting['setting_value'] ?? '';

        // Update setting
        $db->query(
            "UPDATE site_settings SET setting_value = ? WHERE setting_key = ?",
            [$fileUrl, $settingKey]
        );

        // Delete old file if it exists and is local
        if ($oldUrl && strpos($oldUrl, $host) !== false) {
            $oldPath = $uploadDir . basename($oldUrl);
            if (file_exists($oldPath) && is_file($oldPath)) {
                @unlink($oldPath);
            }
        }
    } else {
        // Insert new setting
        $db->query(
            "INSERT INTO site_settings (setting_key, setting_value, setting_type, description, is_public)
             VALUES (?, ?, 'url', ?, TRUE)",
            [$settingKey, $fileUrl, $settingDescriptions[$uploadType] ?? 'Image URL']
        );
    }

    // Log activity
    logActivity('update', 'site_settings', null, "Updated {$uploadType} to {$filename}");

    successResponse([
        'url' => $fileUrl,
        'filename' => $filename,
        'type' => $uploadType
    ], ucfirst(str_replace('_', ' ', $uploadType)) . ' uploaded successfully');

} catch (Exception $e) {
    error_log("Logo upload error: " . $e->getMessage());
    // Delete uploaded file if database update failed
    @unlink($filepath);
    errorResponse('Failed to update database');
}
