<?php
/**
 * Ars Medical Frankfurt - Site Settings API
 * Handles CRUD operations for site settings
 */

if (!defined('SECURE_ACCESS')) {
    define('SECURE_ACCESS', true);
}

require_once __DIR__ . '/../includes/functions.php';

header('Content-Type: application/json');

$method = $_SERVER['REQUEST_METHOD'];

switch ($method) {
    case 'GET':
        handleGet();
        break;
    case 'POST':
        handlePost();
        break;
    case 'PUT':
        handlePut();
        break;
    case 'DELETE':
        handleDelete();
        break;
    default:
        errorResponse('Method not allowed', 405);
}

/**
 * Handle GET requests - Get all settings or single setting
 */
function handleGet() {
    $key = $_GET['key'] ?? null;
    $publicOnly = !isset($_GET['admin']);
    
    try {
        $db = getDB();
        
        if ($key) {
            // Get single setting
            $sql = "SELECT * FROM site_settings WHERE setting_key = ?";
            $params = [$key];
            
            if ($publicOnly) {
                $sql .= " AND is_public = 1";
            }
            
            $setting = $db->fetchOne($sql, $params);
            
            if (!$setting) {
                errorResponse('Setting not found', 404);
            }
            
            successResponse(['setting' => $setting]);
        } else {
            // Get all settings
            $sql = "SELECT * FROM site_settings WHERE 1=1";
            $params = [];
            
            if ($publicOnly) {
                $sql .= " AND is_public = 1";
            }
            
            $sql .= " ORDER BY setting_key ASC";
            
            $settings = $db->fetchAll($sql, $params);
            
            // Convert to key-value array for easier use
            $settingsMap = [];
            foreach ($settings as $setting) {
                $settingsMap[$setting['setting_key']] = $setting;
            }
            
            successResponse([
                'settings' => $settings,
                'settings_map' => $settingsMap
            ]);
        }
    } catch (Exception $e) {
        error_log("Settings GET error: " . $e->getMessage());
        errorResponse('Failed to retrieve settings');
    }
}

/**
 * Handle POST requests - Create new setting (admin only)
 */
function handlePost() {
    // Require admin authentication
    requireAdmin();
    
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        errorResponse('Invalid JSON data', 400);
    }
    
    $key = sanitizeInput($data['setting_key'] ?? '');
    $value = $data['setting_value'] ?? '';
    $type = sanitizeInput($data['setting_type'] ?? 'text');
    $description = sanitizeInput($data['description'] ?? '');
    $isPublic = isset($data['is_public']) ? (bool)$data['is_public'] : true;
    
    // Validate required fields
    if (empty($key)) {
        errorResponse('Setting key is required');
    }
    
    // Validate setting type
    $validTypes = ['text', 'textarea', 'email', 'phone', 'url', 'number', 'boolean'];
    if (!in_array($type, $validTypes)) {
        errorResponse('Invalid setting type');
    }
    
    try {
        $db = getDB();
        
        // Check if key already exists
        $existing = $db->fetchOne(
            "SELECT id FROM site_settings WHERE setting_key = ?",
            [$key]
        );
        
        if ($existing) {
            errorResponse('Setting key already exists');
        }
        
        // Insert new setting
        $db->query(
            "INSERT INTO site_settings (setting_key, setting_value, setting_type, description, is_public) 
             VALUES (?, ?, ?, ?, ?)",
            [$key, $value, $type, $description, $isPublic ? 1 : 0]
        );
        
        $newId = $db->getConnection()->lastInsertId();
        
        // Log activity
        logActivity('create', 'site_settings', $newId, "Created setting: {$key}");
        
        successResponse([
            'id' => $newId,
            'setting_key' => $key,
            'setting_value' => $value
        ], 'Setting created successfully');
        
    } catch (Exception $e) {
        error_log("Settings POST error: " . $e->getMessage());
        errorResponse('Failed to create setting');
    }
}

/**
 * Handle PUT requests - Update setting (admin only)
 */
function handlePut() {
    // Require admin authentication
    requireAdmin();
    
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        errorResponse('Invalid JSON data', 400);
    }
    
    $key = sanitizeInput($data['setting_key'] ?? '');
    $value = $data['setting_value'] ?? null;
    $type = isset($data['setting_type']) ? sanitizeInput($data['setting_type']) : null;
    $description = isset($data['description']) ? sanitizeInput($data['description']) : null;
    $isPublic = isset($data['is_public']) ? (bool)$data['is_public'] : null;
    
    if (empty($key)) {
        errorResponse('Setting key is required');
    }
    
    try {
        $db = getDB();
        
        // Check if setting exists
        $existing = $db->fetchOne(
            "SELECT id FROM site_settings WHERE setting_key = ?",
            [$key]
        );
        
        if (!$existing) {
            errorResponse('Setting not found', 404);
        }
        
        // Build update query dynamically
        $updates = [];
        $params = [];
        
        if ($value !== null) {
            $updates[] = "setting_value = ?";
            $params[] = $value;
        }
        
        if ($type !== null) {
            $validTypes = ['text', 'textarea', 'email', 'phone', 'url', 'number', 'boolean'];
            if (!in_array($type, $validTypes)) {
                errorResponse('Invalid setting type');
            }
            $updates[] = "setting_type = ?";
            $params[] = $type;
        }
        
        if ($description !== null) {
            $updates[] = "description = ?";
            $params[] = $description;
        }
        
        if ($isPublic !== null) {
            $updates[] = "is_public = ?";
            $params[] = $isPublic ? 1 : 0;
        }
        
        if (empty($updates)) {
            errorResponse('No fields to update');
        }
        
        $params[] = $key;
        
        $sql = "UPDATE site_settings SET " . implode(', ', $updates) . " WHERE setting_key = ?";
        $db->query($sql, $params);
        
        // Log activity
        logActivity('update', 'site_settings', $existing['id'], "Updated setting: {$key}");
        
        successResponse(['setting_key' => $key], 'Setting updated successfully');
        
    } catch (Exception $e) {
        error_log("Settings PUT error: " . $e->getMessage());
        errorResponse('Failed to update setting');
    }
}

/**
 * Handle DELETE requests - Delete setting (admin only)
 */
function handleDelete() {
    // Require admin authentication
    requireAdmin();
    
    $key = sanitizeInput($_GET['key'] ?? '');
    
    if (empty($key)) {
        errorResponse('Setting key is required');
    }
    
    try {
        $db = getDB();
        
        // Check if setting exists
        $existing = $db->fetchOne(
            "SELECT id FROM site_settings WHERE setting_key = ?",
            [$key]
        );
        
        if (!$existing) {
            errorResponse('Setting not found', 404);
        }
        
        // Delete setting
        $db->query("DELETE FROM site_settings WHERE setting_key = ?", [$key]);
        
        // Log activity
        logActivity('delete', 'site_settings', $existing['id'], "Deleted setting: {$key}");
        
        successResponse(null, 'Setting deleted successfully');
        
    } catch (Exception $e) {
        error_log("Settings DELETE error: " . $e->getMessage());
        errorResponse('Failed to delete setting');
    }
}
