<?php
/**
 * Ars Medical Frankfurt - Patients API
 * CRUD operations for patient management
 */

if (!defined('SECURE_ACCESS')) {
    define('SECURE_ACCESS', true);
}

require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/notifications.php';

// Require authentication for all operations
requireAuth();

header('Content-Type: application/json');

$method = $_SERVER['REQUEST_METHOD'];

switch ($method) {
    case 'GET':
        handleGet();
        break;
    case 'POST':
        handlePost();
        break;
    case 'PUT':
        handlePut();
        break;
    case 'DELETE':
        handleDelete();
        break;
    default:
        errorResponse('Method not allowed', 405);
}

/**
 * Handle GET requests - List or get single patient
 */
function handleGet() {
    $id = isset($_GET['id']) ? intval($_GET['id']) : null;

    try {
        $db = getDB();

        // Check if patients table exists
        $tableExists = $db->fetchOne("SHOW TABLES LIKE 'patients'");
        if (!$tableExists) {
            // Return empty array if table doesn't exist
            successResponse([]);
            return;
        }

        if ($id) {
            // Get single patient
            $patient = $db->fetchOne("SELECT * FROM patients WHERE id = ?", [$id]);

            if (!$patient) {
                errorResponse('Patient not found', 404);
            }

            successResponse($patient);
        } else {
            // List patients with filters
            $filters = [
                'status' => sanitizeInput($_GET['status'] ?? ''),
                'department' => sanitizeInput($_GET['department'] ?? ''),
                'search' => sanitizeInput($_GET['search'] ?? '')
            ];

            // Remove empty filters
            $filters = array_filter($filters);

            $patients = getPatients($filters);
            successResponse($patients);
        }
    } catch (Exception $e) {
        error_log("Get patients error: " . $e->getMessage());
        errorResponse('Failed to retrieve patients');
    }
}

/**
 * Handle POST requests - Create new patient
 */
function handlePost() {
    // Validate CSRF token
    $headers = getallheaders();
    $csrfToken = $headers['X-CSRF-Token'] ?? $_POST['csrf_token'] ?? '';
    
    if (!validateCSRFToken($csrfToken)) {
        errorResponse('Invalid security token', 403);
    }
    
    // Get POST data
    $data = json_decode(file_get_contents('php://input'), true);
    if (!$data) {
        $data = $_POST;
    }
    
    // Validate required fields
    $required = ['firstName', 'lastName', 'birthDate', 'street', 'postalCode', 'city', 
                 'phone', 'emergencyContact', 'emergencyPhone', 'diagnosis', 
                 'department', 'doctor', 'roomNumber', 'admissionDate', 'insurance', 'insuranceNumber'];
    
    foreach ($required as $field) {
        if (empty($data[$field])) {
            errorResponse("Field '$field' is required");
        }
    }
    
    try {
        $db = getDB();
        
        // Calculate age from birth date
        $age = calculateAge($data['birthDate']);
        
        $sql = "INSERT INTO patients (
            first_name, last_name, birth_date, age, street, postal_code, city,
            phone, email, emergency_contact, emergency_phone, diagnosis,
            department, doctor, room_number, blood_group, admission_date,
            allergies, conditions, insurance, insurance_number, insurance_type,
            status, created_by
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $params = [
            sanitizeInput($data['firstName']),
            sanitizeInput($data['lastName']),
            $data['birthDate'],
            $age,
            sanitizeInput($data['street']),
            sanitizeInput($data['postalCode']),
            sanitizeInput($data['city']),
            sanitizeInput($data['phone']),
            !empty($data['email']) ? sanitizeInput($data['email']) : null,
            sanitizeInput($data['emergencyContact']),
            sanitizeInput($data['emergencyPhone']),
            sanitizeInput($data['diagnosis']),
            sanitizeInput($data['department']),
            sanitizeInput($data['doctor']),
            sanitizeInput($data['roomNumber']),
            !empty($data['bloodGroup']) ? sanitizeInput($data['bloodGroup']) : null,
            $data['admissionDate'],
            !empty($data['allergies']) ? sanitizeInput($data['allergies']) : null,
            !empty($data['conditions']) ? sanitizeInput($data['conditions']) : null,
            sanitizeInput($data['insurance']),
            sanitizeInput($data['insuranceNumber']),
            !empty($data['insuranceType']) ? sanitizeInput($data['insuranceType']) : 'gesetzlich',
            'aufgenommen',
            getCurrentUserId()
        ];
        
        $patientId = $db->insert($sql, $params);
        
        // Log activity
        logActivity('create', 'patients', $patientId, 'Patient created: ' . $data['firstName'] . ' ' . $data['lastName']);
        
        // Send notification to emergency contact if enabled
        $notificationResult = null;
        if (!empty($data['notifyEmergencyContact']) && $data['notifyEmergencyContact'] === 'true') {
            $patientData = [
                'first_name' => $data['firstName'],
                'last_name' => $data['lastName'],
                'emergency_contact' => $data['emergencyContact'],
                'emergency_phone' => $data['emergencyPhone'],
                'emergency_email' => $data['emergencyEmail'] ?? null,
                'admission_date' => $data['admissionDate'],
                'department' => $data['department'],
                'room_number' => $data['roomNumber']
            ];
            
            $notificationType = $data['notificationType'] ?? 'both';
            $notificationResult = sendEmergencyContactNotification($patientId, $patientData, $notificationType);
        }
        
        $response = ['id' => $patientId];
        if ($notificationResult) {
            $response['notification'] = $notificationResult;
        }
        
        successResponse($response, 'Patient created successfully');
        
    } catch (Exception $e) {
        error_log("Create patient error: " . $e->getMessage());
        errorResponse('Failed to create patient');
    }
}

/**
 * Handle PUT requests - Update patient
 */
function handlePut() {
    // Validate CSRF token
    $headers = getallheaders();
    $csrfToken = $headers['X-CSRF-Token'] ?? '';
    
    if (!validateCSRFToken($csrfToken)) {
        errorResponse('Invalid security token', 403);
    }
    
    // Get PUT data
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data || empty($data['id'])) {
        errorResponse('Patient ID is required');
    }
    
    $id = intval($data['id']);
    
    try {
        $db = getDB();
        
        // Check if patient exists
        $existing = $db->fetchOne("SELECT id FROM patients WHERE id = ?", [$id]);
        if (!$existing) {
            errorResponse('Patient not found', 404);
        }
        
        // Build update query dynamically
        $fields = [];
        $params = [];
        
        $fieldMap = [
            'firstName' => 'first_name',
            'lastName' => 'last_name',
            'birthDate' => 'birth_date',
            'street' => 'street',
            'postalCode' => 'postal_code',
            'city' => 'city',
            'phone' => 'phone',
            'email' => 'email',
            'emergencyContact' => 'emergency_contact',
            'emergencyPhone' => 'emergency_phone',
            'diagnosis' => 'diagnosis',
            'department' => 'department',
            'doctor' => 'doctor',
            'roomNumber' => 'room_number',
            'bloodGroup' => 'blood_group',
            'admissionDate' => 'admission_date',
            'allergies' => 'allergies',
            'conditions' => 'conditions',
            'insurance' => 'insurance',
            'insuranceNumber' => 'insurance_number',
            'insuranceType' => 'insurance_type',
            'status' => 'status'
        ];
        
        foreach ($fieldMap as $inputName => $dbField) {
            if (isset($data[$inputName])) {
                $fields[] = "$dbField = ?";
                $params[] = sanitizeInput($data[$inputName]);
            }
        }
        
        // Recalculate age if birth date changed
        if (isset($data['birthDate'])) {
            $fields[] = "age = ?";
            $params[] = calculateAge($data['birthDate']);
        }
        
        if (empty($fields)) {
            errorResponse('No fields to update');
        }
        
        $params[] = $id;
        $sql = "UPDATE patients SET " . implode(', ', $fields) . " WHERE id = ?";
        
        $db->update($sql, $params);
        
        // Log activity
        logActivity('update', 'patients', $id, 'Patient updated');
        
        successResponse(null, 'Patient updated successfully');
        
    } catch (Exception $e) {
        error_log("Update patient error: " . $e->getMessage());
        errorResponse('Failed to update patient');
    }
}

/**
 * Handle DELETE requests - Delete patient
 */
function handleDelete() {
    // Validate CSRF token
    $headers = getallheaders();
    $csrfToken = $headers['X-CSRF-Token'] ?? '';
    
    if (!validateCSRFToken($csrfToken)) {
        errorResponse('Invalid security token', 403);
    }
    
    $id = isset($_GET['id']) ? intval($_GET['id']) : null;
    
    if (!$id) {
        errorResponse('Patient ID is required');
    }
    
    try {
        $db = getDB();
        
        // Check if patient exists
        $patient = $db->fetchOne("SELECT first_name, last_name FROM patients WHERE id = ?", [$id]);
        if (!$patient) {
            errorResponse('Patient not found', 404);
        }
        
        // Delete patient (invoices will be deleted via CASCADE)
        $db->delete("DELETE FROM patients WHERE id = ?", [$id]);
        
        // Log activity
        logActivity('delete', 'patients', $id, 'Patient deleted: ' . $patient['first_name'] . ' ' . $patient['last_name']);
        
        successResponse(null, 'Patient deleted successfully');
        
    } catch (Exception $e) {
        error_log("Delete patient error: " . $e->getMessage());
        errorResponse('Failed to delete patient');
    }
}
