<?php
/**
 * Ars Medical Frankfurt - Invoices API
 * CRUD operations for invoice management
 */

// Enable error logging
error_reporting(E_ALL);
ini_set('display_errors', '0');
ini_set('log_errors', '1');

if (!defined('SECURE_ACCESS')) {
    define('SECURE_ACCESS', true);
}

require_once __DIR__ . '/../includes/functions.php';

// Require authentication
requireAuth();

header('Content-Type: application/json');

$method = $_SERVER['REQUEST_METHOD'];

switch ($method) {
    case 'GET':
        handleGet();
        break;
    case 'POST':
        handlePost();
        break;
    case 'PUT':
        handlePut();
        break;
    case 'DELETE':
        handleDelete();
        break;
    default:
        errorResponse('Method not allowed', 405);
}

/**
 * Handle GET requests
 */
function handleGet() {
    $id = isset($_GET['id']) ? intval($_GET['id']) : null;

    try {
        $db = getDB();

        // Check if invoices table exists
        $tableExists = $db->fetchOne("SHOW TABLES LIKE 'invoices'");
        if (!$tableExists) {
            // Return empty array if table doesn't exist
            successResponse([]);
            return;
        }

        if ($id) {
            // Get single invoice
            $invoice = $db->fetchOne("SELECT * FROM invoices WHERE id = ?", [$id]);

            if (!$invoice) {
                errorResponse('Invoice not found', 404);
            }

            successResponse($invoice);
        } else {
            // List invoices with filters
            $filters = [
                'status' => sanitizeInput($_GET['status'] ?? ''),
                'patient_id' => isset($_GET['patient_id']) ? intval($_GET['patient_id']) : null,
                'search' => sanitizeInput($_GET['search'] ?? '')
            ];

            $filters = array_filter($filters);
            $invoices = getInvoices($filters);
            successResponse($invoices);
        }
    } catch (Exception $e) {
        error_log("Get invoices error: " . $e->getMessage());
        errorResponse('Failed to retrieve invoices: ' . $e->getMessage());
    }
}

/**
 * Handle POST requests - Create invoice
 */
function handlePost() {
    // Validate CSRF token
    $headers = getallheaders();
    $csrfToken = $headers['X-CSRF-Token'] ?? $_POST['csrf_token'] ?? '';
    
    if (!validateCSRFToken($csrfToken)) {
        errorResponse('Invalid security token', 403);
    }
    
    // Get data
    $data = json_decode(file_get_contents('php://input'), true);
    if (!$data) {
        $data = $_POST;
    }
    
    // Validate required fields
    if (empty($data['patientId'])) {
        errorResponse('Patient ID is required');
    }
    
    try {
        $db = getDB();

        // Check if invoices table exists
        $tableExists = $db->fetchOne("SHOW TABLES LIKE 'invoices'");
        if (!$tableExists) {
            errorResponse('Invoice system not initialized. Please run database setup.', 500);
        }

        // Get patient info
        $patient = $db->fetchOne("SELECT first_name, last_name FROM patients WHERE id = ?", [$data['patientId']]);
        if (!$patient) {
            errorResponse('Patient not found', 404);
        }
        
        // Calculate totals
        $treatmentCost = floatval($data['treatmentCost'] ?? 0);
        $medicationCost = floatval($data['medicationCost'] ?? 0);
        $roomCostPerDay = floatval($data['roomCostPerDay'] ?? 0);
        $stayDays = intval($data['stayDays'] ?? 1);
        $doctorFee = floatval($data['doctorFee'] ?? 0);
        $otherCosts = floatval($data['otherCosts'] ?? 0);
        
        $roomTotal = $roomCostPerDay * $stayDays;
        $subtotal = $treatmentCost + $medicationCost + $roomTotal + $doctorFee + $otherCosts;
        $taxRate = 19.00;
        $taxAmount = $subtotal * ($taxRate / 100);
        $totalAmount = $subtotal + $taxAmount;
        
        // Generate invoice number
        $invoiceNumber = generateInvoiceNumber();
        
        $sql = "INSERT INTO invoices (
            invoice_number, patient_id, patient_name, invoice_date, due_date,
            treatment_cost, medication_cost, room_cost_per_day, stay_days,
            doctor_fee, other_costs, subtotal, tax_amount, tax_rate, total_amount,
            status, notes, created_by
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $params = [
            $invoiceNumber,
            $data['patientId'],
            $patient['first_name'] . ' ' . $patient['last_name'],
            $data['invoiceDate'] ?? date('Y-m-d'),
            $data['dueDate'] ?? date('Y-m-d', strtotime('+14 days')),
            $treatmentCost,
            $medicationCost,
            $roomCostPerDay,
            $stayDays,
            $doctorFee,
            $otherCosts,
            $subtotal,
            $taxAmount,
            $taxRate,
            $totalAmount,
            'offen',
            !empty($data['notes']) ? sanitizeInput($data['notes']) : null,
            getCurrentUserId()
        ];
        
        $invoiceId = $db->insert($sql, $params);
        
        // Log activity
        logActivity('create', 'invoices', $invoiceId, 'Invoice created: ' . $invoiceNumber);
        
        successResponse([
            'id' => $invoiceId,
            'invoice_number' => $invoiceNumber
        ], 'Invoice created successfully');
        
    } catch (Exception $e) {
        error_log("Create invoice error: " . $e->getMessage());
        errorResponse('Failed to create invoice');
    }
}

/**
 * Handle PUT requests - Update invoice
 */
function handlePut() {
    // Validate CSRF token
    $headers = getallheaders();
    $csrfToken = $headers['X-CSRF-Token'] ?? '';

    if (!validateCSRFToken($csrfToken)) {
        errorResponse('Invalid security token', 403);
    }

    // Check for action parameter (e.g., mark_paid)
    $action = $_GET['action'] ?? '';
    $id = isset($_GET['id']) ? intval($_GET['id']) : null;

    if ($action === 'mark_paid' && $id) {
        markInvoiceAsPaid($id);
        return;
    }

    $data = json_decode(file_get_contents('php://input'), true);

    if (!$data || empty($data['id'])) {
        errorResponse('Invoice ID is required');
    }

    $id = intval($data['id']);
    
    try {
        $db = getDB();
        
        // Check if invoice exists
        $existing = $db->fetchOne("SELECT id FROM invoices WHERE id = ?", [$id]);
        if (!$existing) {
            errorResponse('Invoice not found', 404);
        }
        
        // If status is being updated to 'bezahlt', set paid_date
        if (isset($data['status']) && $data['status'] === 'bezahlt') {
            $data['paidDate'] = date('Y-m-d');
        }
        
        $fields = [];
        $params = [];
        
        $fieldMap = [
            'status' => 'status',
            'paidDate' => 'paid_date',
            'notes' => 'notes'
        ];
        
        foreach ($fieldMap as $inputName => $dbField) {
            if (isset($data[$inputName])) {
                $fields[] = "$dbField = ?";
                $params[] = sanitizeInput($data[$inputName]);
            }
        }
        
        if (empty($fields)) {
            errorResponse('No fields to update');
        }
        
        $params[] = $id;
        $sql = "UPDATE invoices SET " . implode(', ', $fields) . " WHERE id = ?";
        
        $db->update($sql, $params);
        
        // Log activity
        logActivity('update', 'invoices', $id, 'Invoice updated');
        
        successResponse(null, 'Invoice updated successfully');

    } catch (Exception $e) {
        error_log("Update invoice error: " . $e->getMessage());
        errorResponse('Failed to update invoice');
    }
}

/**
 * Mark invoice as paid
 */
function markInvoiceAsPaid($id) {
    try {
        $db = getDB();

        // Check if invoice exists
        $existing = $db->fetchOne("SELECT id, status FROM invoices WHERE id = ?", [$id]);
        if (!$existing) {
            errorResponse('Invoice not found', 404);
        }

        // Update invoice status to paid
        $db->update(
            "UPDATE invoices SET status = 'bezahlt', paid_date = CURDATE() WHERE id = ?",
            [$id]
        );

        // Log activity
        logActivity('update', 'invoices', $id, 'Invoice marked as paid');

        successResponse(null, 'Invoice marked as paid successfully');

    } catch (Exception $e) {
        error_log("Mark paid error: " . $e->getMessage());
        errorResponse('Failed to mark invoice as paid');
    }
}

/**
 * Handle DELETE requests
 */
function handleDelete() {
    // Validate CSRF token
    $headers = getallheaders();
    $csrfToken = $headers['X-CSRF-Token'] ?? '';
    
    if (!validateCSRFToken($csrfToken)) {
        errorResponse('Invalid security token', 403);
    }
    
    $id = isset($_GET['id']) ? intval($_GET['id']) : null;
    
    if (!$id) {
        errorResponse('Invoice ID is required');
    }
    
    try {
        $db = getDB();
        
        // Check if invoice exists
        $invoice = $db->fetchOne("SELECT invoice_number FROM invoices WHERE id = ?", [$id]);
        if (!$invoice) {
            errorResponse('Invoice not found', 404);
        }
        
        $db->delete("DELETE FROM invoices WHERE id = ?", [$id]);
        
        // Log activity
        logActivity('delete', 'invoices', $id, 'Invoice deleted: ' . $invoice['invoice_number']);
        
        successResponse(null, 'Invoice deleted successfully');
        
    } catch (Exception $e) {
        error_log("Delete invoice error: " . $e->getMessage());
        errorResponse('Failed to delete invoice');
    }
}
