<?php
/**
 * Ars Medical Frankfurt - Database Setup Script
 * Creates missing tables and columns
 */

if (!defined('SECURE_ACCESS')) {
    define('SECURE_ACCESS', true);
}

require_once __DIR__ . '/../includes/functions.php';

header('Content-Type: application/json');

// Only allow admin access
requireAuth();

$results = [
    'success' => true,
    'actions' => [],
    'errors' => []
];

try {
    $db = getDB();

    // Create patients table if not exists
    try {
        $db->query("CREATE TABLE IF NOT EXISTS patients (
            id INT AUTO_INCREMENT PRIMARY KEY,
            first_name VARCHAR(100) NOT NULL,
            last_name VARCHAR(100) NOT NULL,
            birth_date DATE NOT NULL,
            age INT NOT NULL,
            street VARCHAR(255) NOT NULL,
            postal_code VARCHAR(10) NOT NULL,
            city VARCHAR(100) NOT NULL,
            phone VARCHAR(20) NOT NULL,
            email VARCHAR(100) NULL,
            emergency_contact VARCHAR(100) NOT NULL,
            emergency_phone VARCHAR(20) NOT NULL,
            diagnosis TEXT NOT NULL,
            department VARCHAR(50) NOT NULL,
            doctor VARCHAR(100) NOT NULL,
            room_number VARCHAR(20) NOT NULL,
            blood_group VARCHAR(5) NULL,
            admission_date DATE NOT NULL,
            allergies TEXT NULL,
            conditions TEXT NULL,
            insurance VARCHAR(50) NOT NULL,
            insurance_number VARCHAR(50) NOT NULL,
            insurance_type ENUM('gesetzlich', 'privat', 'beihilfe') DEFAULT 'gesetzlich',
            status ENUM('aufgenommen', 'entlassen') DEFAULT 'aufgenommen',
            discharge_date DATE NULL,
            created_by INT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL,
            INDEX idx_name (last_name, first_name),
            INDEX idx_status (status),
            INDEX idx_department (department),
            INDEX idx_admission_date (admission_date)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
        $results['actions'][] = 'Created patients table';
    } catch (Exception $e) {
        $results['errors'][] = 'Error creating patients table: ' . $e->getMessage();
    }

    // Create invoices table if not exists
    try {
        $db->query("CREATE TABLE IF NOT EXISTS invoices (
            id INT AUTO_INCREMENT PRIMARY KEY,
            invoice_number VARCHAR(50) NOT NULL UNIQUE,
            patient_id INT NOT NULL,
            patient_name VARCHAR(200) NOT NULL,
            invoice_date DATE NOT NULL,
            due_date DATE NOT NULL,
            treatment_cost DECIMAL(10, 2) DEFAULT 0.00,
            medication_cost DECIMAL(10, 2) DEFAULT 0.00,
            room_cost_per_day DECIMAL(10, 2) DEFAULT 0.00,
            stay_days INT DEFAULT 1,
            doctor_fee DECIMAL(10, 2) DEFAULT 0.00,
            other_costs DECIMAL(10, 2) DEFAULT 0.00,
            subtotal DECIMAL(10, 2) NOT NULL,
            tax_amount DECIMAL(10, 2) NOT NULL,
            tax_rate DECIMAL(5, 2) DEFAULT 19.00,
            total_amount DECIMAL(10, 2) NOT NULL,
            status ENUM('offen', 'bezahlt', 'ueberfaellig', 'storniert') DEFAULT 'offen',
            paid_date DATE NULL,
            notes TEXT NULL,
            created_by INT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            FOREIGN KEY (patient_id) REFERENCES patients(id) ON DELETE CASCADE,
            FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL,
            INDEX idx_invoice_number (invoice_number),
            INDEX idx_status (status),
            INDEX idx_invoice_date (invoice_date),
            INDEX idx_patient_id (patient_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
        $results['actions'][] = 'Created invoices table';
    } catch (Exception $e) {
        $results['errors'][] = 'Error creating invoices table: ' . $e->getMessage();
    }

    // Create site_settings table if not exists
    try {
        $db->query("CREATE TABLE IF NOT EXISTS site_settings (
            id INT AUTO_INCREMENT PRIMARY KEY,
            setting_key VARCHAR(100) NOT NULL UNIQUE,
            setting_value TEXT NULL,
            setting_type ENUM('text', 'textarea', 'email', 'phone', 'url', 'number', 'boolean') DEFAULT 'text',
            description VARCHAR(255) NULL,
            is_public BOOLEAN DEFAULT TRUE,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            INDEX idx_setting_key (setting_key),
            INDEX idx_is_public (is_public)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
        $results['actions'][] = 'Created site_settings table';
    } catch (Exception $e) {
        $results['errors'][] = 'Error creating site_settings table: ' . $e->getMessage();
    }

    // Create notifications table if not exists
    try {
        $db->query("CREATE TABLE IF NOT EXISTS notifications (
            id INT AUTO_INCREMENT PRIMARY KEY,
            patient_id INT NOT NULL,
            notification_type ENUM('email', 'sms', 'both') NOT NULL,
            recipient_name VARCHAR(100) NOT NULL,
            recipient_email VARCHAR(100) NULL,
            recipient_phone VARCHAR(20) NULL,
            subject VARCHAR(255) NULL,
            message TEXT NOT NULL,
            status ENUM('pending', 'sent', 'failed') DEFAULT 'pending',
            sent_at TIMESTAMP NULL,
            error_message TEXT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (patient_id) REFERENCES patients(id) ON DELETE CASCADE,
            INDEX idx_patient_id (patient_id),
            INDEX idx_status (status),
            INDEX idx_created_at (created_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
        $results['actions'][] = 'Created notifications table';
    } catch (Exception $e) {
        $results['errors'][] = 'Error creating notifications table: ' . $e->getMessage();
    }

    // Insert default settings if they don't exist
    $defaultSettings = [
        ['hospital_name', 'Ars Medical Frankfurt', 'text', 'Hospital name displayed across the site', true],
        ['hospital_phone', '+49 69 1234 5678', 'phone', 'Main hospital phone number', true],
        ['hospital_email', 'info@arsmedicafrankfurt.de', 'email', 'Main hospital email address', true],
        ['email_from_address', 'noreply@arsmedicafrankfurt.de', 'email', 'Default from email address', false],
        ['email_from_name', 'Ars Medical Frankfurt', 'text', 'Default from name', false],
        ['email_use_smtp', '0', 'boolean', 'Use SMTP for emails (1=yes, 0=no)', false],
        ['sms_provider', '', 'text', 'SMS provider (twilio, vonage, sms77, or empty)', false],
        ['sms_api_key', '', 'text', 'SMS API Key', false],
    ];

    foreach ($defaultSettings as $setting) {
        try {
            $exists = $db->fetchOne("SELECT id FROM site_settings WHERE setting_key = ?", [$setting[0]]);
            if (!$exists) {
                $db->insert(
                    "INSERT INTO site_settings (setting_key, setting_value, setting_type, description, is_public) VALUES (?, ?, ?, ?, ?)",
                    $setting
                );
                $results['actions'][] = 'Added setting: ' . $setting[0];
            }
        } catch (Exception $e) {
            $results['errors'][] = 'Error adding setting ' . $setting[0] . ': ' . $e->getMessage();
        }
    }

    if (!empty($results['errors'])) {
        $results['success'] = false;
    }

    echo json_encode($results, JSON_PRETTY_PRINT);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Database setup failed: ' . $e->getMessage(),
        'actions' => $results['actions'],
        'errors' => array_merge($results['errors'], [$e->getMessage()])
    ], JSON_PRETTY_PRINT);
}
