<?php
/**
 * Ars Medical Frankfurt - Database Check Script
 * Diagnoses database connection and table structure issues
 */

if (!defined('SECURE_ACCESS')) {
    define('SECURE_ACCESS', true);
}

require_once __DIR__ . '/../includes/functions.php';

header('Content-Type: application/json');

// Only allow admin access
requireAuth();

$results = [
    'success' => true,
    'connection' => false,
    'tables' => [],
    'errors' => []
];

try {
    // Test database connection
    $db = getDB();
    $results['connection'] = true;

    // Check required tables
    $requiredTables = [
        'users' => 'Admin authentication',
        'patients' => 'Patient records',
        'invoices' => 'Invoice records',
        'site_settings' => 'Global configuration',
        'notifications' => 'Notification tracking',
        'activity_log' => 'Audit trail',
        'login_attempts' => 'Security tracking'
    ];

    foreach ($requiredTables as $table => $description) {
        try {
            $tableExists = $db->fetchOne("SHOW TABLES LIKE ?", [$table]);
            if ($tableExists) {
                // Get row count
                $count = $db->fetchOne("SELECT COUNT(*) as count FROM {$table}");
                $results['tables'][$table] = [
                    'exists' => true,
                    'description' => $description,
                    'rows' => $count['count']
                ];
            } else {
                $results['tables'][$table] = [
                    'exists' => false,
                    'description' => $description,
                    'error' => 'Table does not exist'
                ];
                $results['errors'][] = "Missing table: {$table}";
            }
        } catch (Exception $e) {
            $results['tables'][$table] = [
                'exists' => false,
                'description' => $description,
                'error' => $e->getMessage()
            ];
            $results['errors'][] = "Error checking {$table}: " . $e->getMessage();
        }
    }

    // Check specific columns in patients table
    if ($results['tables']['patients']['exists'] ?? false) {
        try {
            $columns = $db->fetchAll("SHOW COLUMNS FROM patients");
            $columnNames = array_column($columns, 'Field');
            $requiredColumns = [
                'id', 'first_name', 'last_name', 'birth_date', 'age', 'street',
                'postal_code', 'city', 'phone', 'email', 'emergency_contact',
                'emergency_phone', 'diagnosis', 'department', 'doctor', 'room_number',
                'blood_group', 'admission_date', 'allergies', 'conditions',
                'insurance', 'insurance_number', 'insurance_type', 'status',
                'discharge_date', 'created_by', 'created_at', 'updated_at'
            ];
            $missingColumns = array_diff($requiredColumns, $columnNames);
            if (!empty($missingColumns)) {
                $results['tables']['patients']['missing_columns'] = $missingColumns;
                $results['errors'][] = "Patients table missing columns: " . implode(', ', $missingColumns);
            }
        } catch (Exception $e) {
            $results['errors'][] = "Error checking patients columns: " . $e->getMessage();
        }
    }

    // Check specific columns in invoices table
    if ($results['tables']['invoices']['exists'] ?? false) {
        try {
            $columns = $db->fetchAll("SHOW COLUMNS FROM invoices");
            $columnNames = array_column($columns, 'Field');
            $requiredColumns = [
                'id', 'invoice_number', 'patient_id', 'patient_name', 'invoice_date',
                'due_date', 'treatment_cost', 'medication_cost', 'room_cost_per_day',
                'stay_days', 'doctor_fee', 'other_costs', 'subtotal', 'tax_amount',
                'tax_rate', 'total_amount', 'status', 'paid_date', 'notes',
                'created_by', 'created_at', 'updated_at'
            ];
            $missingColumns = array_diff($requiredColumns, $columnNames);
            if (!empty($missingColumns)) {
                $results['tables']['invoices']['missing_columns'] = $missingColumns;
                $results['errors'][] = "Invoices table missing columns: " . implode(', ', $missingColumns);
            }
        } catch (Exception $e) {
            $results['errors'][] = "Error checking invoices columns: " . $e->getMessage();
        }
    }

    // Check site_settings for email/SMS config
    if ($results['tables']['site_settings']['exists'] ?? false) {
        try {
            $settings = $db->fetchAll("SELECT setting_key FROM site_settings WHERE setting_key LIKE 'email_%' OR setting_key LIKE 'sms_%'");
            $results['notification_settings'] = array_column($settings, 'setting_key');
        } catch (Exception $e) {
            $results['errors'][] = "Error checking notification settings: " . $e->getMessage();
        }
    }

    if (!empty($results['errors'])) {
        $results['success'] = false;
    }

    echo json_encode($results, JSON_PRETTY_PRINT);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'connection' => false,
        'error' => 'Database connection failed: ' . $e->getMessage(),
        'tables' => [],
        'errors' => [$e->getMessage()]
    ], JSON_PRETTY_PRINT);
}
