<?php
/**
 * Ars Medical Frankfurt - Dashboard API
 * Statistics and overview data
 */

// Enable error reporting for debugging (remove in production)
error_reporting(E_ALL);
ini_set('display_errors', '0');
ini_set('log_errors', '1');

if (!defined('SECURE_ACCESS')) {
    define('SECURE_ACCESS', true);
}

try {
    require_once __DIR__ . '/../includes/functions.php';
} catch (Exception $e) {
    http_response_code(500);
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'error' => 'Failed to load functions: ' . $e->getMessage()]);
    exit;
}

// Require authentication
try {
    requireAuth();
} catch (Exception $e) {
    http_response_code(401);
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'error' => 'Authentication failed: ' . $e->getMessage()]);
    exit;
}

header('Content-Type: application/json');

$action = $_GET['action'] ?? '';

switch ($action) {
    case 'stats':
        getDashboardStats();
        break;
    case 'recent_patients':
        getRecentPatients();
        break;
    case 'recent_invoices':
        getRecentInvoices();
        break;
    case 'department_stats':
        getDepartmentStats();
        break;
    default:
        // Return all dashboard data
        getFullDashboard();
}

/**
 * Get full dashboard data
 */
function getFullDashboard() {
    try {
        $data = [
            'stats' => getDashboardStatsData(),
            'recent_patients' => getRecentPatientsData(),
            'recent_invoices' => getRecentInvoicesData()
        ];
        
        successResponse($data);
    } catch (Exception $e) {
        error_log("Dashboard error: " . $e->getMessage());
        errorResponse('Failed to load dashboard data: ' . $e->getMessage());
    }
}

/**
 * Get dashboard statistics
 */
function getDashboardStats() {
    try {
        $stats = getDashboardStatsData();
        successResponse($stats);
    } catch (Exception $e) {
        error_log("Stats error: " . $e->getMessage());
        errorResponse('Failed to load statistics');
    }
}

/**
 * Get stats data
 */
function getDashboardStatsData() {
    $db = getDB();

    // Check if tables exist
    $patientsExists = $db->fetchOne("SHOW TABLES LIKE 'patients'");
    $invoicesExists = $db->fetchOne("SHOW TABLES LIKE 'invoices'");

    // Total admitted patients
    if ($patientsExists) {
        $result = $db->fetchOne("SELECT COUNT(*) as count FROM patients WHERE status = 'aufgenommen'");
        $totalPatients = $result['count'];

        // Today's admissions
        $result = $db->fetchOne("SELECT COUNT(*) as count FROM patients WHERE DATE(admission_date) = CURDATE()");
        $todayAdmissions = $result['count'];
    } else {
        $totalPatients = 0;
        $todayAdmissions = 0;
    }

    // Invoice stats
    if ($invoicesExists) {
        // Open invoices
        $result = $db->fetchOne("SELECT COUNT(*) as count FROM invoices WHERE status = 'offen'");
        $openInvoices = $result['count'];

        // Monthly revenue
        $result = $db->fetchOne("SELECT COALESCE(SUM(total_amount), 0) as total FROM invoices WHERE MONTH(invoice_date) = MONTH(CURDATE()) AND YEAR(invoice_date) = YEAR(CURDATE())");
        $monthlyRevenue = $result['total'];

        // Overdue invoices
        $result = $db->fetchOne("SELECT COUNT(*) as count FROM invoices WHERE status = 'offen' AND due_date < CURDATE()");
        $overdueInvoices = $result['count'];

        // Total revenue (all time)
        $result = $db->fetchOne("SELECT COALESCE(SUM(total_amount), 0) as total FROM invoices WHERE status = 'bezahlt'");
        $totalRevenue = $result['total'];
    } else {
        $openInvoices = 0;
        $monthlyRevenue = 0;
        $overdueInvoices = 0;
        $totalRevenue = 0;
    }

    return [
        'total_patients' => intval($totalPatients),
        'today_admissions' => intval($todayAdmissions),
        'open_invoices' => intval($openInvoices),
        'monthly_revenue' => floatval($monthlyRevenue),
        'overdue_invoices' => intval($overdueInvoices),
        'total_revenue' => floatval($totalRevenue)
    ];
}

/**
 * Get recent patients
 */
function getRecentPatients() {
    try {
        $patients = getRecentPatientsData();
        successResponse($patients);
    } catch (Exception $e) {
        error_log("Recent patients error: " . $e->getMessage());
        errorResponse('Failed to load recent patients');
    }
}

/**
 * Get recent patients data
 */
function getRecentPatientsData($limit = 5) {
    $db = getDB();

    // Check if table exists
    $tableExists = $db->fetchOne("SHOW TABLES LIKE 'patients'");
    if (!$tableExists) {
        return [];
    }

    return $db->fetchAll(
        "SELECT id, first_name, last_name, room_number, doctor, admission_date, status, department
         FROM patients
         ORDER BY created_at DESC
         LIMIT ?",
        [$limit]
    );
}

/**
 * Get recent invoices
 */
function getRecentInvoices() {
    try {
        $invoices = getRecentInvoicesData();
        successResponse($invoices);
    } catch (Exception $e) {
        error_log("Recent invoices error: " . $e->getMessage());
        errorResponse('Failed to load recent invoices');
    }
}

/**
 * Get recent invoices data
 */
function getRecentInvoicesData($limit = 5) {
    $db = getDB();

    // Check if table exists
    $tableExists = $db->fetchOne("SHOW TABLES LIKE 'invoices'");
    if (!$tableExists) {
        return [];
    }

    return $db->fetchAll(
        "SELECT id, invoice_number, patient_name, invoice_date, due_date, total_amount, status
         FROM invoices
         ORDER BY created_at DESC
         LIMIT ?",
        [$limit]
    );
}

/**
 * Get department statistics
 */
function getDepartmentStats() {
    try {
        $db = getDB();

        // Check if table exists
        $tableExists = $db->fetchOne("SHOW TABLES LIKE 'patients'");
        if (!$tableExists) {
            successResponse([]);
            return;
        }

        $stats = $db->fetchAll(
            "SELECT department, COUNT(*) as patient_count
             FROM patients
             WHERE status = 'aufgenommen'
             GROUP BY department
             ORDER BY patient_count DESC"
        );

        successResponse($stats);
    } catch (Exception $e) {
        error_log("Department stats error: " . $e->getMessage());
        errorResponse('Failed to load department statistics');
    }
}
