<?php
/**
 * Ars Medical Frankfurt - Configuration File
 * 
 * IMPORTANT: Change these settings for production use!
 * Make sure to use strong passwords and keep this file secure.
 */

// Allow inclusion from other files
if (!defined('SECURE_ACCESS')) {
    define('SECURE_ACCESS', true);
}

// Database Configuration
// TODO: Update these with your actual database credentials
define('DB_HOST', 'localhost');
define('DB_NAME', 'arsmedica_db');
define('DB_USER', 'arsmedica_user');
define('DB_PASS', 'your_secure_password_here');
define('DB_CHARSET', 'utf8mb4');

// Application Settings
define('APP_NAME', 'Ars Medical Frankfurt');
define('APP_URL', 'https://arsmedicafrankfurt.de'); // Update with your domain
define('ADMIN_EMAIL', 'admin@arsmedicafrankfurt.de');

// Session Configuration
define('SESSION_NAME', 'arsmedica_session');
define('SESSION_TIMEOUT', 1800); // 30 minutes in seconds
define('SESSION_REGENERATE', 300); // Regenerate ID every 5 minutes

// Security Settings
define('CSRF_TOKEN_NAME', 'csrf_token');
define('MAX_LOGIN_ATTEMPTS', 5); // Max failed login attempts
define('LOGIN_LOCKOUT_TIME', 900); // 15 minutes lockout

// Error Reporting
// Set to false in production!
define('DEBUG_MODE', false);

if (DEBUG_MODE) {
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
} else {
    error_reporting(0);
    ini_set('display_errors', 0);
    ini_set('log_errors', 1);
    ini_set('error_log', __DIR__ . '/../logs/error.log');
}

// Timezone
date_default_timezone_set('Europe/Berlin');

// Security Headers Function
function setSecurityHeaders() {
    // Prevent clickjacking
    header('X-Frame-Options: DENY');
    
    // XSS Protection
    header('X-XSS-Protection: 1; mode=block');
    
    // Prevent MIME type sniffing
    header('X-Content-Type-Options: nosniff');
    
    // Referrer Policy
    header('Referrer-Policy: strict-origin-when-cross-origin');
    
    // Content Security Policy (basic)
    header("Content-Security-Policy: default-src 'self'; script-src 'self' 'unsafe-inline' https://fonts.googleapis.com; style-src 'self' 'unsafe-inline' https://fonts.googleapis.com https://fonts.gstatic.com; font-src 'self' https://fonts.gstatic.com; img-src 'self' data: https:; connect-src 'self';");
    
    // HSTS (HTTPS Strict Transport Security) - Enable only with HTTPS
    // header('Strict-Transport-Security: max-age=31536000; includeSubDomains; preload');
}

// Call security headers
setSecurityHeaders();

// Session Configuration
ini_set('session.cookie_httponly', 1);
// Only enable secure cookies on HTTPS (production)
ini_set('session.cookie_secure', isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 1 : 0);
ini_set('session.cookie_samesite', 'Strict');
ini_set('session.use_strict_mode', 1);
ini_set('session.gc_maxlifetime', SESSION_TIMEOUT);

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_name(SESSION_NAME);
    session_start();
}

// Regenerate session ID periodically for security
if (isset($_SESSION['last_regeneration'])) {
    if (time() - $_SESSION['last_regeneration'] > SESSION_REGENERATE) {
        session_regenerate_id(true);
        $_SESSION['last_regeneration'] = time();
    }
} else {
    $_SESSION['last_regeneration'] = time();
}

// Check session timeout (but don't redirect if already on login page)
$currentPath = $_SERVER['REQUEST_URI'] ?? '';
$isLoginPage = strpos($currentPath, '/admin/login.php') !== false;

if (isset($_SESSION['last_activity'])) {
    if (time() - $_SESSION['last_activity'] > SESSION_TIMEOUT) {
        // Session expired
        session_unset();
        session_destroy();
        if (!$isLoginPage) {
            header('Location: /admin/login.php?error=timeout');
            exit;
        }
    }
}

// Only update last_activity if not on login page (to prevent session creation on login page)
if (!$isLoginPage) {
    $_SESSION['last_activity'] = time();
}

/**
 * Generate CSRF Token
 */
function generateCSRFToken() {
    if (empty($_SESSION[CSRF_TOKEN_NAME])) {
        $_SESSION[CSRF_TOKEN_NAME] = bin2hex(random_bytes(32));
    }
    return $_SESSION[CSRF_TOKEN_NAME];
}

/**
 * Validate CSRF Token
 */
function validateCSRFToken($token) {
    return isset($_SESSION[CSRF_TOKEN_NAME]) && hash_equals($_SESSION[CSRF_TOKEN_NAME], $token);
}

/**
 * Sanitize Input
 */
function sanitizeInput($data) {
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
    return $data;
}

/**
 * Output escaping for HTML
 */
function e($string) {
    return htmlspecialchars($string, ENT_QUOTES, 'UTF-8');
}

/**
 * Log Activity
 */
function logActivity($action, $entityType = null, $entityId = null, $details = null) {
    try {
        $db = Database::getInstance();
        $userId = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : null;
        $ipAddress = $_SERVER['REMOTE_ADDR'] ?? null;
        
        $stmt = $db->prepare("INSERT INTO activity_log (user_id, action, entity_type, entity_id, details, ip_address) VALUES (?, ?, ?, ?, ?, ?)");
        $stmt->execute([$userId, $action, $entityType, $entityId, $details, $ipAddress]);
    } catch (Exception $e) {
        // Silently fail - don't expose errors
        error_log("Activity log error: " . $e->getMessage());
    }
}

/**
 * Check Rate Limiting for Login
 */
function checkRateLimit($username, $ipAddress) {
    try {
        $db = Database::getInstance();
        $lockoutTime = date('Y-m-d H:i:s', time() - LOGIN_LOCKOUT_TIME);
        
        // Check recent failed attempts
        $stmt = $db->prepare("SELECT COUNT(*) as attempts FROM login_attempts WHERE (username = ? OR ip_address = ?) AND attempted_at > ? AND success = 0");
        $stmt->execute([$username, $ipAddress, $lockoutTime]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return $result['attempts'] < MAX_LOGIN_ATTEMPTS;
    } catch (Exception $e) {
        return true; // Allow if we can't check
    }
}

/**
 * Record Login Attempt
 */
function recordLoginAttempt($username, $ipAddress, $success) {
    try {
        $db = Database::getInstance();
        $stmt = $db->prepare("INSERT INTO login_attempts (username, ip_address, success) VALUES (?, ?, ?)");
        $stmt->execute([$username, $ipAddress, $success ? 1 : 0]);
    } catch (Exception $e) {
        error_log("Login attempt recording error: " . $e->getMessage());
    }
}
