<?php
/**
 * Ars Medical Frankfurt - Authentication API
 * Handles login/logout operations
 */

if (!defined('SECURE_ACCESS')) {
    define('SECURE_ACCESS', true);
}

require_once __DIR__ . '/../includes/functions.php';

header('Content-Type: application/json');

$action = $_POST['action'] ?? '';

switch ($action) {
    case 'login':
        handleLogin();
        break;
    case 'logout':
        handleLogout();
        break;
    case 'check':
        checkAuth();
        break;
    default:
        errorResponse('Invalid action', 400);
}

/**
 * Handle login
 */
function handleLogin() {
    $username = sanitizeInput($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';
    $csrfToken = $_POST['csrf_token'] ?? '';
    
    // Validate CSRF token
    if (!validateCSRFToken($csrfToken)) {
        errorResponse('Invalid security token', 403);
    }
    
    // Validate input
    if (empty($username) || empty($password)) {
        errorResponse('Username and password are required');
    }
    
    $ipAddress = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    
    // Check rate limiting
    if (!checkRateLimit($username, $ipAddress)) {
        recordLoginAttempt($username, $ipAddress, false);
        errorResponse('Too many failed attempts. Please try again later.', 429);
    }
    
    try {
        $db = getDB();
        
        // Get user by username
        $user = $db->fetchOne(
            "SELECT id, username, password_hash, email, full_name, role, is_active FROM users WHERE username = ?",
            [$username]
        );
        
        // Check if user exists and is active
        if (!$user || !$user['is_active']) {
            recordLoginAttempt($username, $ipAddress, false);
            errorResponse('Invalid username or password');
        }
        
        // Verify password
        if (!password_verify($password, $user['password_hash'])) {
            recordLoginAttempt($username, $ipAddress, false);
            errorResponse('Invalid username or password');
        }
        
        // Check if password needs rehash (for algorithm upgrades)
        if (password_needs_rehash($user['password_hash'], PASSWORD_BCRYPT)) {
            $newHash = password_hash($password, PASSWORD_BCRYPT);
            $db->update("UPDATE users SET password_hash = ? WHERE id = ?", [$newHash, $user['id']]);
        }
        
        // Regenerate session ID to prevent session fixation
        session_regenerate_id(true);
        
        // Set session variables
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['username'] = $user['username'];
        $_SESSION['email'] = $user['email'];
        $_SESSION['full_name'] = $user['full_name'];
        $_SESSION['user_role'] = $user['role'];
        $_SESSION['last_activity'] = time();
        $_SESSION['last_regeneration'] = time();
        
        // Update last login
        $db->update("UPDATE users SET last_login = NOW() WHERE id = ?", [$user['id']]);
        
        // Record successful login
        recordLoginAttempt($username, $ipAddress, true);
        
        // Log activity
        logActivity('login', 'users', $user['id'], 'User logged in');
        
        successResponse([
            'user' => [
                'id' => $user['id'],
                'username' => $user['username'],
                'email' => $user['email'],
                'full_name' => $user['full_name'],
                'role' => $user['role']
            ]
        ], 'Login successful');
        
    } catch (Exception $e) {
        error_log("Login error: " . $e->getMessage());
        errorResponse('An error occurred during login');
    }
}

/**
 * Handle logout
 */
function handleLogout() {
    if (isLoggedIn()) {
        // Log activity before destroying session
        logActivity('logout', 'users', getCurrentUserId(), 'User logged out');
        
        // Clear session data
        $_SESSION = [];
        
        // Destroy session cookie
        if (isset($_COOKIE[session_name()])) {
            setcookie(session_name(), '', [
                'expires' => time() - 3600,
                'path' => '/',
                'secure' => true,
                'httponly' => true,
                'samesite' => 'Strict'
            ]);
        }
        
        // Destroy session
        session_destroy();
    }
    
    successResponse(null, 'Logout successful');
}

/**
 * Check authentication status
 */
function checkAuth() {
    if (isLoggedIn()) {
        $user = getCurrentUser();
        successResponse([
            'authenticated' => true,
            'user' => $user
        ]);
    } else {
        successResponse([
            'authenticated' => false,
            'user' => null
        ]);
    }
}
