<?php
/**
 * Ars Medical Frankfurt - Site Settings Admin Page
 */

if (!defined('SECURE_ACCESS')) {
    define('SECURE_ACCESS', true);
}

require_once __DIR__ . '/../includes/functions.php';

// Require admin authentication
requireAdmin();

// Get all settings (including private ones for admin)
$settings = getAllSettings(true);

// Get logo URLs
$logoUrl = getSetting('logo_url', '');
$faviconUrl = getSetting('logo_favicon', '');
$footerImageUrl = getSetting('footer_image_url', '');

// Group settings by category
$groupedSettings = [
    'Logo & Branding' => [],
    'Kontakt' => [],
    'Soziale Medien' => [],
    'SEO & Meta' => [],
    'E-Mail Konfiguration' => [],
    'SMS Konfiguration' => [],
    'Sonstiges' => []
];

foreach ($settings as $setting) {
    $key = $setting['setting_key'];
    if ($key === 'logo_url' || $key === 'logo_favicon') {
        // Skip logo settings - handled separately
        continue;
    } elseif (strpos($key, 'email_') === 0) {
        $groupedSettings['E-Mail Konfiguration'][] = $setting;
    } elseif (strpos($key, 'sms_') === 0) {
        $groupedSettings['SMS Konfiguration'][] = $setting;
    } elseif (strpos($key, 'phone') !== false || strpos($key, 'email') !== false || strpos($key, 'address') !== false || strpos($key, 'hours') !== false || $key === 'hospital_name') {
        $groupedSettings['Kontakt'][] = $setting;
    } elseif (strpos($key, '_url') !== false) {
        $groupedSettings['Soziale Medien'][] = $setting;
    } elseif (strpos($key, 'meta_') !== false) {
        $groupedSettings['SEO & Meta'][] = $setting;
    } else {
        $groupedSettings['Sonstiges'][] = $setting;
    }
}

$pageTitle = 'Einstellungen | Admin Dashboard';
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $pageTitle; ?></title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary: #004B7C;
            --primary-dark: #003a61;
            --primary-light: #0066a8;
            --secondary: #00A896;
            --accent: #E8F4F8;
            --danger: #DC2626;
            --success: #10B981;
            --warning: #F59E0B;
            --white: #FFFFFF;
            --gray-50: #F8FAFC;
            --gray-100: #F1F5F9;
            --gray-200: #E2E8F0;
            --gray-300: #CBD5E1;
            --gray-400: #94A3B8;
            --gray-500: #64748B;
            --gray-600: #475569;
            --gray-700: #334155;
            --gray-800: #1E293B;
            --gray-900: #0F172A;
        }

        * { margin: 0; padding: 0; box-sizing: border-box; }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: var(--gray-100);
            color: var(--gray-800);
            line-height: 1.6;
        }

        /* Layout */
        .admin-container {
            display: flex;
            min-height: 100vh;
        }

        /* Sidebar */
        .sidebar {
            width: 260px;
            background: var(--white);
            border-right: 1px solid var(--gray-200);
            display: flex;
            flex-direction: column;
            position: fixed;
            height: 100vh;
            z-index: 100;
        }

        .sidebar-header {
            padding: 1.5rem;
            border-bottom: 1px solid var(--gray-200);
        }

        .sidebar-header a {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            text-decoration: none;
        }

        .sidebar-header .logo-icon {
            width: 40px;
            height: 40px;
            background: var(--primary);
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .sidebar-header .logo-icon svg {
            width: 24px;
            height: 24px;
            color: var(--white);
        }

        .sidebar-header .logo-text h1 {
            font-size: 1.125rem;
            font-weight: 700;
            color: var(--gray-900);
            line-height: 1.2;
        }

        .sidebar-header .logo-text span {
            font-size: 0.6875rem;
            color: var(--gray-500);
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }

        .sidebar-nav {
            flex: 1;
            padding: 1rem 0.75rem;
            overflow-y: auto;
        }

        .sidebar-section {
            margin-bottom: 1.5rem;
        }

        .sidebar-section-title {
            font-size: 0.6875rem;
            font-weight: 600;
            color: var(--gray-400);
            text-transform: uppercase;
            letter-spacing: 0.1em;
            padding: 0 0.75rem;
            margin-bottom: 0.5rem;
        }

        .sidebar-nav a {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem;
            color: var(--gray-600);
            text-decoration: none;
            font-size: 0.9375rem;
            font-weight: 500;
            border-radius: 8px;
            transition: all 0.2s;
            margin-bottom: 0.25rem;
        }

        .sidebar-nav a:hover {
            background: var(--gray-100);
            color: var(--gray-900);
        }

        .sidebar-nav a.active {
            background: var(--accent);
            color: var(--primary);
        }

        .sidebar-nav a svg {
            width: 20px;
            height: 20px;
        }

        .sidebar-footer {
            padding: 1rem 0.75rem;
            border-top: 1px solid var(--gray-200);
        }

        .sidebar-footer .user-menu {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem;
            cursor: pointer;
            border-radius: 8px;
            transition: background 0.2s;
        }

        .sidebar-footer .user-menu:hover {
            background: var(--gray-100);
        }

        .user-avatar {
            width: 36px;
            height: 36px;
            background: var(--primary);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: var(--white);
            font-weight: 600;
            font-size: 0.875rem;
        }

        .user-info {
            flex: 1;
        }

        .user-info strong {
            display: block;
            font-size: 0.875rem;
            color: var(--gray-900);
        }

        .user-info span {
            font-size: 0.75rem;
            color: var(--gray-500);
        }

        .logout-btn {
            background: none;
            border: none;
            cursor: pointer;
            padding: 0.5rem;
            color: var(--gray-400);
            transition: color 0.2s;
        }

        .logout-btn:hover {
            color: var(--danger);
        }

        /* Main Content */
        .main-content {
            flex: 1;
            margin-left: 260px;
            padding: 2rem;
        }

        /* Header */
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
        }

        .page-header h1 {
            font-size: 1.875rem;
            font-weight: 700;
            color: var(--gray-900);
        }

        .user-menu {
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .user-menu span {
            color: var(--gray-600);
        }

        .btn {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.625rem 1.25rem;
            border-radius: 6px;
            font-size: 0.875rem;
            font-weight: 500;
            text-decoration: none;
            cursor: pointer;
            transition: all 0.2s;
            border: none;
        }

        .btn-primary {
            background: var(--primary);
            color: var(--white);
        }

        .btn-primary:hover {
            background: var(--primary-dark);
        }

        .btn-secondary {
            background: var(--gray-200);
            color: var(--gray-700);
        }

        .btn-secondary:hover {
            background: var(--gray-300);
        }

        /* Settings Grid */
        .settings-container {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(500px, 1fr));
            gap: 1.5rem;
        }

        @media (max-width: 768px) {
            .settings-container {
                grid-template-columns: 1fr;
            }
        }

        .settings-group {
            background: var(--white);
            border-radius: 16px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.08), 0 4px 12px rgba(0,0,0,0.05);
            overflow: hidden;
            border: 1px solid var(--gray-200);
            transition: box-shadow 0.2s;
        }

        .settings-group:hover {
            box-shadow: 0 4px 12px rgba(0,0,0,0.1), 0 8px 24px rgba(0,0,0,0.08);
        }

        .settings-group-header {
            padding: 1.25rem 1.5rem;
            background: linear-gradient(135deg, var(--gray-50) 0%, var(--white) 100%);
            border-bottom: 1px solid var(--gray-200);
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .settings-group-icon {
            width: 40px;
            height: 40px;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            background: var(--primary);
            color: var(--white);
        }

        .settings-group-icon svg {
            width: 20px;
            height: 20px;
        }

        .settings-group-title {
            flex: 1;
        }

        .settings-group-header h2 {
            font-size: 1.125rem;
            font-weight: 600;
            color: var(--gray-800);
        }

        .settings-group-header p {
            font-size: 0.8125rem;
            color: var(--gray-500);
            margin-top: 0.125rem;
        }

        .settings-group-content {
            padding: 1.5rem;
        }

        .setting-item {
            margin-bottom: 1.5rem;
            padding-bottom: 1.5rem;
            border-bottom: 1px solid var(--gray-100);
        }

        .setting-item:last-child {
            margin-bottom: 0;
            padding-bottom: 0;
            border-bottom: none;
        }

        .setting-label {
            display: block;
            font-weight: 500;
            color: var(--gray-700);
            margin-bottom: 0.375rem;
            font-size: 0.9375rem;
        }

        .setting-key {
            font-size: 0.6875rem;
            color: var(--gray-400);
            font-family: 'SF Mono', Monaco, monospace;
            background: var(--gray-100);
            padding: 0.125rem 0.375rem;
            border-radius: 4px;
            margin-left: 0.5rem;
            font-weight: 500;
        }

        .setting-description {
            font-size: 0.8125rem;
            color: var(--gray-500);
            margin-bottom: 0.625rem;
            line-height: 1.5;
        }

        .setting-input {
            width: 100%;
            padding: 0.625rem 0.875rem;
            border: 1px solid var(--gray-300);
            border-radius: 8px;
            font-size: 0.9375rem;
            font-family: inherit;
            transition: all 0.2s;
            background: var(--white);
        }

        .setting-input:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(0, 75, 124, 0.1);
        }

        .setting-input:hover {
            border-color: var(--gray-400);
        }

        textarea.setting-input {
            min-height: 80px;
            resize: vertical;
        }

        .setting-actions {
            display: flex;
            gap: 0.75rem;
            margin-top: 0.75rem;
        }

        .btn-sm {
            padding: 0.375rem 0.75rem;
            font-size: 0.8125rem;
        }

        /* Provider Selection Cards */
        .provider-grid {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 0.75rem;
            margin-bottom: 1rem;
        }

        .provider-card {
            padding: 1rem;
            border: 2px solid var(--gray-200);
            border-radius: 10px;
            text-align: center;
            cursor: pointer;
            transition: all 0.2s;
        }

        .provider-card:hover {
            border-color: var(--primary-light);
        }

        .provider-card.active {
            border-color: var(--primary);
            background: var(--accent);
        }

        .provider-card svg {
            width: 32px;
            height: 32px;
            margin-bottom: 0.5rem;
            color: var(--gray-600);
        }

        .provider-card.active svg {
            color: var(--primary);
        }

        .provider-card span {
            display: block;
            font-size: 0.875rem;
            font-weight: 500;
            color: var(--gray-700);
        }

        /* Status Badge */
        .status-badge {
            display: inline-flex;
            align-items: center;
            gap: 0.375rem;
            padding: 0.25rem 0.75rem;
            border-radius: 9999px;
            font-size: 0.75rem;
            font-weight: 500;
        }

        .status-badge.success {
            background: #D1FAE5;
            color: #065F46;
        }

        .status-badge.warning {
            background: #FEF3C7;
            color: #92400E;
        }

        /* Alert */
        .alert {
            padding: 1rem 1.25rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
            display: none;
        }

        .alert.show {
            display: block;
        }

        .alert-success {
            background: #D1FAE5;
            border: 1px solid #A7F3D0;
            color: #065F46;
        }

        .alert-error {
            background: #FEE2E2;
            border: 1px solid #FECACA;
            color: #991B1B;
        }

        /* Responsive */
        @media (max-width: 1024px) {
            .sidebar {
                transform: translateX(-100%);
                transition: transform 0.3s;
            }

            .sidebar.active {
                transform: translateX(0);
            }

            .main-content {
                margin-left: 0;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Sidebar -->
        <aside class="sidebar">
            <div class="sidebar-header">
                <a href="index.php">
                    <div class="logo-icon">
                        <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"/>
                        </svg>
                    </div>
                    <div class="logo-text">
                        <h1>Ars Medical</h1>
                        <span>Admin Dashboard</span>
                    </div>
                </a>
            </div>
            <nav class="sidebar-nav">
                <div class="sidebar-section">
                    <div class="sidebar-section-title">Hauptmenü</div>
                    <a href="index.php">
                        <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2H6a2 2 0 01-2-2V6zM14 6a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2h-2a2 2 0 01-2-2V6zM4 16a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2H6a2 2 0 01-2-2v-2zM14 16a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2h-2a2 2 0 01-2-2v-2z"/></svg>
                        Dashboard
                    </a>
                    <a href="patients.php">
                        <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z"/></svg>
                        Patienten
                    </a>
                    <a href="invoices.php">
                        <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/></svg>
                        Rechnungen
                    </a>
                </div>
                <div class="sidebar-section">
                    <div class="sidebar-section-title">System</div>
                    <a href="settings.php" class="active">
                        <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z"/><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/></svg>
                        Einstellungen
                    </a>
                    <a href="logout.php">
                        <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"/></svg>
                        Abmelden
                    </a>
                </div>
            </nav>
            <div class="sidebar-footer">
                <div class="user-menu">
                    <div class="user-avatar"><?php echo substr($_SESSION['full_name'] ?? 'AD', 0, 2); ?></div>
                    <div class="user-info">
                        <strong><?php echo e($_SESSION['full_name'] ?? 'Admin'); ?></strong>
                        <span><?php echo e($_SESSION['email'] ?? 'admin@arsmedica.de'); ?></span>
                    </div>
                    <a href="logout.php" class="logout-btn" title="Abmelden">
                        <svg width="20" height="20" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"/></svg>
                    </a>
                </div>
            </div>
        </aside>

        <!-- Main Content -->
        <main class="main-content">
            <div class="page-header">
                <h1>Website Einstellungen</h1>
            </div>

            <div id="alertSuccess" class="alert alert-success"></div>
            <div id="alertError" class="alert alert-error"></div>

            <div class="settings-container">
                <!-- Logo Upload Section -->
                <div class="settings-group">
                    <div class="settings-group-header">
                        <div class="settings-group-icon">
                            <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"/></svg>
                        </div>
                        <div class="settings-group-title">
                            <h2>Logo & Branding</h2>
                            <p>Hospital logo and favicon settings</p>
                        </div>
                    </div>
                    <div class="settings-group-content">
                        <!-- Main Logo Upload -->
                        <div class="setting-item">
                            <label class="setting-label">Hospital Logo</label>
                            <p class="setting-description">Upload your hospital logo (recommended: 200x60px, PNG or SVG)</p>
                            
                            <?php if ($logoUrl): ?>
                                <div class="logo-preview" style="margin-bottom: 1rem; padding: 1rem; background: var(--gray-100); border-radius: 8px;">
                                    <img src="<?php echo e($logoUrl); ?>" alt="Current Logo" style="max-height: 60px; max-width: 200px;">
                                </div>
                            <?php endif; ?>
                            
                            <form class="logo-upload-form" data-type="logo">
                                <input type="file" name="logo_file" accept=".png,.jpg,.jpeg,.svg,.webp" style="margin-bottom: 0.75rem;">
                                <div class="setting-actions">
                                    <button type="submit" class="btn btn-primary btn-sm">
                                        <svg width="14" height="14" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-8l-4-4m0 0L8 8m4-4v12"/></svg>
                                        Logo hochladen
                                    </button>
                                </div>
                            </form>
                        </div>
                        
                        <!-- Favicon Upload -->
                        <div class="setting-item">
                            <label class="setting-label">Favicon</label>
                            <p class="setting-description">Upload favicon for browser tabs (recommended: 32x32px or 64x64px PNG)</p>

                            <?php if ($faviconUrl): ?>
                                <div class="logo-preview" style="margin-bottom: 1rem; padding: 1rem; background: var(--gray-100); border-radius: 8px;">
                                    <img src="<?php echo e($faviconUrl); ?>" alt="Current Favicon" style="height: 32px; width: 32px;">
                                </div>
                            <?php endif; ?>

                            <form class="logo-upload-form" data-type="favicon">
                                <input type="file" name="favicon_file" accept=".png,.jpg,.jpeg,.svg,.webp,.ico" style="margin-bottom: 0.75rem;">
                                <div class="setting-actions">
                                    <button type="submit" class="btn btn-primary btn-sm">
                                        <svg width="14" height="14" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-8l-4-4m0 0L8 8m4-4v12"/></svg>
                                        Favicon hochladen
                                    </button>
                                </div>
                            </form>
                        </div>

                        <!-- Footer Image Upload -->
                        <div class="setting-item">
                            <label class="setting-label">Footer Image</label>
                            <p class="setting-description">Upload footer image/logo (recommended: 300x100px, PNG or JPG)</p>

                            <?php if ($footerImageUrl): ?>
                                <div class="logo-preview" style="margin-bottom: 1rem; padding: 1rem; background: var(--gray-100); border-radius: 8px;">
                                    <img src="<?php echo e($footerImageUrl); ?>" alt="Current Footer Image" style="max-height: 100px; max-width: 300px;">
                                </div>
                            <?php endif; ?>

                            <form class="logo-upload-form" data-type="footer_image">
                                <input type="file" name="footer_image_file" accept=".png,.jpg,.jpeg,.webp" style="margin-bottom: 0.75rem;">
                                <div class="setting-actions">
                                    <button type="submit" class="btn btn-primary btn-sm">
                                        <svg width="14" height="14" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-8l-4-4m0 0L8 8m4-4v12"/></svg>
                                        Footer Bild hochladen
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>

                <?php
                $groupIcons = [
                    'Kontakt' => '<svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"/></svg>',
                    'Soziale Medien' => '<svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 8h2a2 2 0 012 2v6a2 2 0 01-2 2h-2v4l-4-4H9a1.994 1.994 0 01-1.414-.586m0 0L11 14h4a2 2 0 002-2V6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2v4l.586-.586z"/></svg>',
                    'SEO & Meta' => '<svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 7h.01M7 3h5c.512 0 1.024.195 1.414.586l7 7a2 2 0 010 2.828l-7 7a2 2 0 01-2.828 0l-7-7A1.994 1.994 0 013 12V7a4 4 0 014-4z"/></svg>',
                    'E-Mail Konfiguration' => '<svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"/></svg>',
                    'SMS Konfiguration' => '<svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z"/></svg>',
                    'Sonstiges' => '<svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z"/><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/></svg>'
                ];
                $groupDescriptions = [
                    'Kontakt' => 'Hospital contact information and addresses',
                    'Soziale Medien' => 'Social media profile links',
                    'SEO & Meta' => 'Search engine optimization settings',
                    'E-Mail Konfiguration' => 'SMTP and email sending settings',
                    'SMS Konfiguration' => 'SMS provider API settings',
                    'Sonstiges' => 'Other system settings'
                ];
                ?>
                <?php foreach ($groupedSettings as $groupName => $groupSettings): ?>
                    <?php if (!empty($groupSettings)): ?>
                        <div class="settings-group">
                            <div class="settings-group-header">
                                <div class="settings-group-icon">
                                    <?php echo $groupIcons[$groupName] ?? $groupIcons['Sonstiges']; ?>
                                </div>
                                <div class="settings-group-title">
                                    <h2><?php echo e($groupName); ?></h2>
                                    <p><?php echo e($groupDescriptions[$groupName] ?? ''); ?></p>
                                </div>
                            </div>
                            <div class="settings-group-content">
                                <?php foreach ($groupSettings as $setting): ?>
                                    <div class="setting-item" data-key="<?php echo e($setting['setting_key']); ?>">
                                        <label class="setting-label">
                                            <?php echo e(ucfirst(str_replace('_', ' ', $setting['setting_key']))); ?>
                                            <span class="setting-key"><?php echo e($setting['setting_key']); ?></span>
                                        </label>
                                        <?php if ($setting['description']): ?>
                                            <p class="setting-description"><?php echo e($setting['description']); ?></p>
                                        <?php endif; ?>
                                        
                                        <?php if ($setting['setting_type'] === 'textarea'): ?>
                                            <textarea class="setting-input" name="value" rows="3"><?php echo e($setting['setting_value']); ?></textarea>
                                        <?php else: ?>
                                            <input type="text" class="setting-input" name="value" value="<?php echo e($setting['setting_value']); ?>" placeholder="Wert eingeben...">
                                        <?php endif; ?>
                                        
                                        <div class="setting-actions">
                                            <button class="btn btn-primary btn-sm" onclick="saveSetting('<?php echo e($setting['setting_key']); ?>', this)">
                                                <svg width="14" height="14" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/></svg>
                                                Speichern
                                            </button>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    <?php endif; ?>
                <?php endforeach; ?>
            </div>
        </main>
    </div>

    <script>
        async function saveSetting(key, button) {
            const settingItem = button.closest('.setting-item');
            const input = settingItem.querySelector('.setting-input');
            const value = input.value;
            
            button.disabled = true;
            button.innerHTML = '<svg width="14" height="14" fill="none" stroke="currentColor" viewBox="0 0 24 24" style="animation: spin 1s linear infinite;"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"/></svg> Speichern...';
            
            try {
                const response = await fetch('../api/settings.php', {
                    method: 'PUT',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        setting_key: key,
                        setting_value: value
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showAlert('success', 'Einstellung erfolgreich gespeichert!');
                } else {
                    showAlert('error', data.error || 'Fehler beim Speichern');
                }
            } catch (error) {
                showAlert('error', 'Ein Fehler ist aufgetreten. Bitte versuchen Sie es erneut.');
            } finally {
                button.disabled = false;
                button.innerHTML = '<svg width="14" height="14" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/></svg> Speichern';
            }
        }
        
        function showAlert(type, message) {
            const alertEl = document.getElementById(type === 'success' ? 'alertSuccess' : 'alertError');
            alertEl.textContent = message;
            alertEl.classList.add('show');
            
            setTimeout(() => {
                alertEl.classList.remove('show');
            }, 5000);
        }
        
        // Logo upload form handling
        document.querySelectorAll('.logo-upload-form').forEach(form => {
            form.addEventListener('submit', async function(e) {
                e.preventDefault();
                
                const type = this.dataset.type;
                const fileInput = this.querySelector('input[type="file"]');
                const button = this.querySelector('button[type="submit"]');
                
                if (!fileInput.files || fileInput.files.length === 0) {
                    showAlert('error', 'Bitte wählen Sie eine Datei aus');
                    return;
                }
                
                const formData = new FormData();
                formData.append('file', fileInput.files[0]);
                formData.append('type', type);
                
                button.disabled = true;
                button.innerHTML = '<svg width="14" height="14" fill="none" stroke="currentColor" viewBox="0 0 24 24" style="animation: spin 1s linear infinite;"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"/></svg> Hochladen...';
                
                try {
                    const response = await fetch('../api/upload-logo.php', {
                        method: 'POST',
                        body: formData
                    });
                    
                    const data = await response.json();
                    
                    if (data.success) {
                        showAlert('success', data.message || 'Logo erfolgreich hochgeladen!');
                        // Reload page to show new logo
                        setTimeout(() => {
                            window.location.reload();
                        }, 1500);
                    } else {
                        showAlert('error', data.error || 'Fehler beim Hochladen');
                    }
                } catch (error) {
                    showAlert('error', 'Ein Fehler ist aufgetreten. Bitte versuchen Sie es erneut.');
                } finally {
                    button.disabled = false;
                    button.innerHTML = type === 'logo' 
                        ? '<svg width="14" height="14" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-8l-4-4m0 0L8 8m4-4v12"/></svg> Logo hochladen'
                        : '<svg width="14" height="14" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-8l-4-4m0 0L8 8m4-4v12"/></svg> Favicon hochladen';
                }
            });
        });
    </script>
    
    <style>
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
    </style>
</body>
</html>
