<?php
/**
 * Ars Medical Frankfurt - Patient Management
 * Admin page for managing patients with notification support
 */

if (!defined('SECURE_ACCESS')) {
    define('SECURE_ACCESS', true);
}

require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/notifications.php';

// Require authentication
requireAuth();

// Get current user
$currentUser = getCurrentUser();

// Get departments and doctors for dropdowns
$departments = getDepartments();
$doctors = getDoctors();
$bloodGroups = getBloodGroups();
$insuranceCompanies = getInsuranceCompanies();

// Get notification config
$notificationConfig = getNotificationConfig();
$smsEnabled = !empty($notificationConfig['sms_provider']);
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Patientenverwaltung | Ars Medical Frankfurt</title>
    <meta name="description" content="Patient management for Ars Medical Frankfurt hospital.">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary: #004B7C; --primary-dark: #003a61; --primary-light: #0066a8;
            --secondary: #00A896; --accent: #E8F4F8; --white: #FFFFFF;
            --gray-50: #F8FAFC; --gray-100: #F1F5F9; --gray-200: #E2E8F0;
            --gray-300: #CBD5E1; --gray-400: #94A3B8; --gray-500: #64748B;
            --gray-600: #475569; --gray-700: #334155; --gray-800: #1E293B; --gray-900: #0F172A;
            --danger: #DC2626; --danger-light: #FEE2E2; --warning: #F59E0B;
            --warning-light: #FEF3C7; --success: #10B981; --success-light: #D1FAE5;
            --info: #3B82F6; --info-light: #DBEAFE;
            --shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
            --shadow: 0 1px 3px 0 rgb(0 0 0 / 0.1), 0 1px 2px -1px rgb(0 0 0 / 0.1);
            --shadow-md: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
            --shadow-lg: 0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1);
        }

        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: 'Inter', sans-serif; color: var(--gray-800); background: var(--gray-100); line-height: 1.6; }
        
        /* Sidebar */
        .sidebar { width: 260px; background: var(--white); border-right: 1px solid var(--gray-200); display: flex; flex-direction: column; position: fixed; height: 100vh; z-index: 100; }
        .sidebar-header { padding: 1.5rem; border-bottom: 1px solid var(--gray-200); }
        .sidebar-logo { display: flex; align-items: center; gap: 0.75rem; text-decoration: none; }
        .sidebar-logo-icon { width: 40px; height: 40px; background: var(--primary); border-radius: 10px; display: flex; align-items: center; justify-content: center; }
        .sidebar-logo-icon svg { width: 24px; height: 24px; color: var(--white); }
        .sidebar-logo-text h2 { font-size: 1.125rem; font-weight: 700; color: var(--gray-900); line-height: 1.2; }
        .sidebar-logo-text span { font-size: 0.6875rem; color: var(--gray-500); text-transform: uppercase; letter-spacing: 0.05em; }
        .sidebar-nav { flex: 1; padding: 1rem 0.75rem; overflow-y: auto; }
        .sidebar-section { margin-bottom: 1.5rem; }
        .sidebar-section-title { font-size: 0.6875rem; font-weight: 600; color: var(--gray-400); text-transform: uppercase; letter-spacing: 0.1em; padding: 0 0.75rem; margin-bottom: 0.5rem; }
        .sidebar-nav a { display: flex; align-items: center; gap: 0.75rem; padding: 0.75rem; color: var(--gray-600); text-decoration: none; font-size: 0.9375rem; font-weight: 500; border-radius: 8px; transition: all 0.2s; margin-bottom: 0.25rem; }
        .sidebar-nav a:hover { background: var(--gray-100); color: var(--gray-900); }
        .sidebar-nav a.active { background: var(--accent); color: var(--primary); }
        .sidebar-nav a svg { width: 20px; height: 20px; }
        .sidebar-footer { padding: 1rem 0.75rem; border-top: 1px solid var(--gray-200); }
        .user-menu { display: flex; align-items: center; gap: 0.75rem; padding: 0.75rem; cursor: pointer; border-radius: 8px; transition: background 0.2s; }
        .user-menu:hover { background: var(--gray-100); }
        .user-avatar { width: 36px; height: 36px; background: var(--primary); border-radius: 50%; display: flex; align-items: center; justify-content: center; color: var(--white); font-weight: 600; font-size: 0.875rem; }
        .user-info { flex: 1; }
        .user-info strong { display: block; font-size: 0.875rem; color: var(--gray-900); }
        .user-info span { font-size: 0.75rem; color: var(--gray-500); }
        .logout-btn { background: none; border: none; cursor: pointer; padding: 0.5rem; color: var(--gray-400); transition: color 0.2s; }
        .logout-btn:hover { color: var(--danger); }

        /* Main Content */
        .main-content { flex: 1; margin-left: 260px; padding: 2rem; min-height: 100vh; }
        .page-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 2rem; }
        .page-header h1 { font-size: 1.875rem; font-weight: 700; color: var(--gray-900); }

        /* Cards */
        .card { background: var(--white); border-radius: 12px; box-shadow: var(--shadow-sm); border: 1px solid var(--gray-200); margin-bottom: 1.5rem; }
        .card-header { padding: 1.25rem 1.5rem; border-bottom: 1px solid var(--gray-200); display: flex; justify-content: space-between; align-items: center; }
        .card-header h2 { font-size: 1.125rem; font-weight: 600; color: var(--gray-900); }
        .card-body { padding: 1.5rem; }

        /* Buttons */
        .btn { display: inline-flex; align-items: center; justify-content: center; gap: 0.5rem; padding: 0.625rem 1.25rem; border-radius: 8px; font-weight: 600; font-size: 0.875rem; text-decoration: none; transition: all 0.2s; border: none; cursor: pointer; }
        .btn-primary { background: var(--primary); color: var(--white); }
        .btn-primary:hover { background: var(--primary-dark); }
        .btn-secondary { background: var(--white); color: var(--gray-700); border: 1px solid var(--gray-300); }
        .btn-secondary:hover { background: var(--gray-50); border-color: var(--gray-400); }
        .btn-success { background: var(--success); color: var(--white); }
        .btn-success:hover { background: #059669; }
        .btn-sm { padding: 0.375rem 0.75rem; font-size: 0.8125rem; }

        /* Form Styles */
        .form-grid { display: grid; grid-template-columns: repeat(2, 1fr); gap: 1.5rem; }
        .form-group { display: flex; flex-direction: column; }
        .form-group.full-width { grid-column: 1 / -1; }
        .form-group label { font-size: 0.875rem; font-weight: 500; color: var(--gray-700); margin-bottom: 0.5rem; }
        .form-group label .required { color: var(--danger); }
        .form-group input, .form-group select, .form-group textarea { padding: 0.625rem 0.875rem; border: 1px solid var(--gray-300); border-radius: 8px; font-size: 0.9375rem; transition: all 0.2s; }
        .form-group input:focus, .form-group select:focus, .form-group textarea:focus { outline: none; border-color: var(--primary); box-shadow: 0 0 0 3px var(--accent); }
        .form-group textarea { resize: vertical; min-height: 80px; }
        .form-group small { font-size: 0.75rem; color: var(--gray-500); margin-top: 0.25rem; }

        /* Notification Section */
        .notification-section { background: var(--accent); border: 1px solid var(--primary); border-radius: 12px; padding: 1.5rem; margin-top: 1.5rem; }
        .notification-section h3 { font-size: 1rem; font-weight: 600; color: var(--primary); margin-bottom: 1rem; display: flex; align-items: center; gap: 0.5rem; }
        .notification-section h3 svg { width: 20px; height: 20px; }
        .notification-toggle { display: flex; align-items: center; gap: 1rem; margin-bottom: 1rem; }
        .toggle-switch { position: relative; width: 50px; height: 26px; }
        .toggle-switch input { opacity: 0; width: 0; height: 0; }
        .toggle-slider { position: absolute; cursor: pointer; top: 0; left: 0; right: 0; bottom: 0; background-color: var(--gray-300); transition: .4s; border-radius: 26px; }
        .toggle-slider:before { position: absolute; content: ""; height: 20px; width: 20px; left: 3px; bottom: 3px; background-color: white; transition: .4s; border-radius: 50%; }
        input:checked + .toggle-slider { background-color: var(--primary); }
        input:checked + .toggle-slider:before { transform: translateX(24px); }
        .notification-options { display: none; margin-top: 1rem; padding-top: 1rem; border-top: 1px solid var(--primary); }
        .notification-options.show { display: block; }
        .radio-group { display: flex; gap: 1.5rem; margin-top: 0.5rem; }
        .radio-option { display: flex; align-items: center; gap: 0.5rem; }
        .radio-option input[type="radio"] { width: 18px; height: 18px; accent-color: var(--primary); }
        .radio-option label { font-size: 0.875rem; color: var(--gray-700); margin: 0; }

        /* Alert Messages */
        .alert { padding: 1rem 1.25rem; border-radius: 8px; margin-bottom: 1.5rem; display: flex; align-items: center; gap: 0.75rem; }
        .alert-success { background: var(--success-light); border: 1px solid var(--success); color: #065f46; }
        .alert-error { background: var(--danger-light); border: 1px solid var(--danger); color: #991b1b; }
        .alert-warning { background: var(--warning-light); border: 1px solid var(--warning); color: #92400e; }
        .alert svg { width: 20px; height: 20px; flex-shrink: 0; }

        /* Tables */
        .table-container { overflow-x: auto; }
        table { width: 100%; border-collapse: collapse; }
        th, td { padding: 0.875rem 1rem; text-align: left; border-bottom: 1px solid var(--gray-200); }
        th { font-weight: 600; font-size: 0.75rem; text-transform: uppercase; letter-spacing: 0.05em; color: var(--gray-500); background: var(--gray-50); }
        td { font-size: 0.9375rem; color: var(--gray-700); }
        tr:hover td { background: var(--gray-50); }
        .badge { display: inline-flex; align-items: center; padding: 0.25rem 0.625rem; border-radius: 9999px; font-size: 0.75rem; font-weight: 600; }
        .badge-success { background: var(--success-light); color: var(--success); }
        .badge-warning { background: var(--warning-light); color: var(--warning); }

        /* Modal */
        .modal { display: none; position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.5); z-index: 1000; align-items: center; justify-content: center; }
        .modal.show { display: flex; }
        .modal-content { background: var(--white); border-radius: 12px; width: 90%; max-width: 800px; max-height: 90vh; overflow-y: auto; }
        .modal-header { padding: 1.25rem 1.5rem; border-bottom: 1px solid var(--gray-200); display: flex; justify-content: space-between; align-items: center; }
        .modal-header h2 { font-size: 1.25rem; font-weight: 600; }
        .modal-close { background: none; border: none; cursor: pointer; padding: 0.5rem; color: var(--gray-500); }
        .modal-close:hover { color: var(--gray-700); }
        .modal-body { padding: 1.5rem; }
        .modal-footer { padding: 1rem 1.5rem; border-top: 1px solid var(--gray-200); display: flex; justify-content: flex-end; gap: 0.75rem; }

        @media (max-width: 1024px) {
            .sidebar { transform: translateX(-100%); transition: transform 0.3s; }
            .sidebar.show { transform: translateX(0); }
            .main-content { margin-left: 0; }
            .form-grid { grid-template-columns: 1fr; }
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <aside class="sidebar">
        <div class="sidebar-header">
            <a href="../index.php" class="sidebar-logo">
                <div class="sidebar-logo-icon">
                    <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"/></svg>
                </div>
                <div class="sidebar-logo-text">
                    <h2>Ars Medical</h2>
                    <span>Admin Dashboard</span>
                </div>
            </a>
        </div>
        <nav class="sidebar-nav">
            <div class="sidebar-section">
                <div class="sidebar-section-title">Hauptmenü</div>
                <a href="index.php">
                    <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2H6a2 2 0 01-2-2V6zM14 6a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2h-2a2 2 0 01-2-2V6zM4 16a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2H6a2 2 0 01-2-2v-2zM14 16a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2h-2a2 2 0 01-2-2v-2z"/></svg>
                    Dashboard
                </a>
                <a href="patients.php" class="active">
                    <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z"/></svg>
                    Patienten
                </a>
                <a href="invoices.php">
                    <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/></svg>
                    Rechnungen
                </a>
            </div>
            <div class="sidebar-section">
                <div class="sidebar-section-title">System</div>
                <a href="settings.php">
                    <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z"/><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/></svg>
                    Einstellungen
                </a>
            </div>
        </nav>
        <div class="sidebar-footer">
            <div class="user-menu">
                <div class="user-avatar"><?php echo substr($currentUser['full_name'], 0, 2); ?></div>
                <div class="user-info">
                    <strong><?php echo $currentUser['full_name']; ?></strong>
                    <span><?php echo $currentUser['email']; ?></span>
                </div>
                <a href="logout.php" class="logout-btn" title="Abmelden">
                    <svg width="20" height="20" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"/></svg>
                </a>
            </div>
        </div>
    </aside>

    <!-- Main Content -->
    <main class="main-content">
        <div class="page-header">
            <h1>Patientenverwaltung</h1>
            <button class="btn btn-primary" onclick="openModal()">
                <svg width="16" height="16" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"/></svg>
                Neuer Patient
            </button>
        </div>

        <!-- Alert Container -->
        <div id="alertContainer"></div>

        <!-- Patients Table -->
        <div class="card">
            <div class="card-header">
                <h2>Patientenliste</h2>
                <div style="display: flex; gap: 0.75rem;">
                    <input type="text" id="searchInput" placeholder="Suchen..." class="btn btn-secondary btn-sm" style="width: 200px; text-align: left;">
                    <select id="filterStatus" class="btn btn-secondary btn-sm" style="text-align: left;">
                        <option value="">Alle Status</option>
                        <option value="aufgenommen">Aufgenommen</option>
                        <option value="entlassen">Entlassen</option>
                    </select>
                </div>
            </div>
            <div class="card-body">
                <div class="table-container" id="patientsTable">
                    <p style="text-align: center; color: var(--gray-500); padding: 2rem;">Lade Patienten...</p>
                </div>
            </div>
        </div>
    </main>

    <!-- Add Patient Modal -->
    <div class="modal" id="patientModal">
        <div class="modal-content">
            <div class="modal-header">
                <h2>Neuen Patienten aufnehmen</h2>
                <button class="modal-close" onclick="closeModal()">
                    <svg width="24" height="24" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/></svg>
                </button>
            </div>
            <form id="patientForm">
                <div class="modal-body">
                    <div class="form-grid">
                        <!-- Personal Information -->
                        <div class="form-group">
                            <label>Vorname <span class="required">*</span></label>
                            <input type="text" name="firstName" required>
                        </div>
                        <div class="form-group">
                            <label>Nachname <span class="required">*</span></label>
                            <input type="text" name="lastName" required>
                        </div>
                        <div class="form-group">
                            <label>Geburtsdatum <span class="required">*</span></label>
                            <input type="date" name="birthDate" required>
                        </div>
                        <div class="form-group">
                            <label>Blutgruppe</label>
                            <select name="bloodGroup">
                                <?php foreach ($bloodGroups as $value => $label): ?>
                                    <option value="<?php echo $value; ?>"><?php echo $label; ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <!-- Address -->
                        <div class="form-group full-width">
                            <label>Straße <span class="required">*</span></label>
                            <input type="text" name="street" required>
                        </div>
                        <div class="form-group">
                            <label>PLZ <span class="required">*</span></label>
                            <input type="text" name="postalCode" pattern="\d{5}" required>
                        </div>
                        <div class="form-group">
                            <label>Ort <span class="required">*</span></label>
                            <input type="text" name="city" required>
                        </div>

                        <!-- Contact -->
                        <div class="form-group">
                            <label>Telefon <span class="required">*</span></label>
                            <input type="tel" name="phone" required>
                        </div>
                        <div class="form-group">
                            <label>E-Mail</label>
                            <input type="email" name="email">
                        </div>

                        <!-- Emergency Contact -->
                        <div class="form-group">
                            <label>Notfallkontakt Name <span class="required">*</span></label>
                            <input type="text" name="emergencyContact" required>
                        </div>
                        <div class="form-group">
                            <label>Notfallkontakt Telefon <span class="required">*</span></label>
                            <input type="tel" name="emergencyPhone" required>
                        </div>
                        <div class="form-group">
                            <label>Notfallkontakt E-Mail</label>
                            <input type="email" name="emergencyEmail">
                            <small>Für E-Mail-Benachrichtigungen</small>
                        </div>

                        <!-- Medical Information -->
                        <div class="form-group">
                            <label>Abteilung <span class="required">*</span></label>
                            <select name="department" required>
                                <option value="">Bitte wählen</option>
                                <?php foreach ($departments as $value => $label): ?>
                                    <option value="<?php echo $value; ?>"><?php echo $label; ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Arzt <span class="required">*</span></label>
                            <select name="doctor" required>
                                <option value="">Bitte wählen</option>
                                <?php foreach ($doctors as $doctor): ?>
                                    <option value="<?php echo $doctor['full_name']; ?>"><?php echo $doctor['full_name']; ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Zimmer <span class="required">*</span></label>
                            <input type="text" name="roomNumber" required>
                        </div>
                        <div class="form-group">
                            <label>Aufnahmedatum <span class="required">*</span></label>
                            <input type="date" name="admissionDate" value="<?php echo date('Y-m-d'); ?>" required>
                        </div>

                        <!-- Insurance -->
                        <div class="form-group">
                            <label>Krankenkasse <span class="required">*</span></label>
                            <select name="insurance" required>
                                <option value="">Bitte wählen</option>
                                <?php foreach ($insuranceCompanies as $value => $label): ?>
                                    <option value="<?php echo $value; ?>"><?php echo $label; ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Versicherungsnummer <span class="required">*</span></label>
                            <input type="text" name="insuranceNumber" required>
                        </div>
                        <div class="form-group">
                            <label>Versicherungsart</label>
                            <select name="insuranceType">
                                <option value="gesetzlich">Gesetzlich</option>
                                <option value="privat">Privat</option>
                                <option value="beihilfe">Beihilfe</option>
                            </select>
                        </div>

                        <!-- Medical Details -->
                        <div class="form-group full-width">
                            <label>Diagnose <span class="required">*</span></label>
                            <textarea name="diagnosis" required></textarea>
                        </div>
                        <div class="form-group">
                            <label>Allergien</label>
                            <textarea name="allergies"></textarea>
                        </div>
                        <div class="form-group">
                            <label>Vorerkrankungen</label>
                            <textarea name="conditions"></textarea>
                        </div>
                    </div>

                    <!-- Notification Section -->
                    <div class="notification-section">
                        <h3>
                            <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 17h5l-1.405-1.405A2.032 2.032 0 0118 14.158V11a6.002 6.002 0 00-4-5.659V5a2 2 0 10-4 0v.341C7.67 6.165 6 8.388 6 11v3.159c0 .538-.214 1.055-.595 1.436L4 17h5m6 0v1a3 3 0 11-6 0v-1m6 0H9"/></svg>
                            Notfallkontakt benachrichtigen
                        </h3>
                        <div class="notification-toggle">
                            <label class="toggle-switch">
                                <input type="checkbox" name="notifyEmergencyContact" id="notifyToggle" value="true">
                                <span class="toggle-slider"></span>
                            </label>
                            <span>Benachrichtigung bei Aufnahme senden</span>
                        </div>
                        <div class="notification-options" id="notificationOptions">
                            <p style="margin-bottom: 0.75rem; font-size: 0.875rem; color: var(--gray-700);">
                                Wählen Sie die Benachrichtigungsmethode:
                            </p>
                            <div class="radio-group">
                                <div class="radio-option">
                                    <input type="radio" name="notificationType" value="email" id="notifyEmail" checked>
                                    <label for="notifyEmail">E-Mail</label>
                                </div>
                                <div class="radio-option">
                                    <input type="radio" name="notificationType" value="sms" id="notifySMS" <?php echo !$smsEnabled ? 'disabled' : ''; ?>>
                                    <label for="notifySMS">SMS <?php echo !$smsEnabled ? '(nicht konfiguriert)' : ''; ?></label>
                                </div>
                                <div class="radio-option">
                                    <input type="radio" name="notificationType" value="both" id="notifyBoth" <?php echo !$smsEnabled ? 'disabled' : ''; ?>>
                                    <label for="notifyBoth">E-Mail & SMS <?php echo !$smsEnabled ? '(nicht konfiguriert)' : ''; ?></label>
                                </div>
                            </div>
                            <p style="margin-top: 1rem; font-size: 0.75rem; color: var(--gray-500);">
                                Der Notfallkontakt erhält eine Nachricht mit den Aufnahmedetails.
                            </p>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" onclick="closeModal()">Abbrechen</button>
                    <button type="submit" class="btn btn-primary">
                        <svg width="16" height="16" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/></svg>
                        Patient aufnehmen
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script>
        // CSRF Token
        const csrfToken = '<?php echo generateCSRFToken(); ?>';

        // Toggle notification options
        document.getElementById('notifyToggle').addEventListener('change', function() {
            document.getElementById('notificationOptions').classList.toggle('show', this.checked);
        });

        // Load patients on page load
        document.addEventListener('DOMContentLoaded', loadPatients);

        // Search and filter
        document.getElementById('searchInput').addEventListener('input', debounce(loadPatients, 300));
        document.getElementById('filterStatus').addEventListener('change', loadPatients);

        async function loadPatients() {
            const search = document.getElementById('searchInput').value;
            const status = document.getElementById('filterStatus').value;

            try {
                const params = new URLSearchParams();
                if (search) params.append('search', search);
                if (status) params.append('status', status);

                const response = await fetch(`../api/patients.php?${params}`, {
                    headers: { 'X-CSRF-Token': csrfToken }
                });

                const responseText = await response.text();
                let data;
                try {
                    data = JSON.parse(responseText);
                } catch (e) {
                    console.error('Non-JSON response:', responseText);
                    throw new Error(`Server error: ${response.status}. Check server logs.`);
                }

                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}: ${data.error || response.statusText}`);
                }

                if (data.success) {
                    renderPatientsTable(data.data);
                } else {
                    showAlert(data.error || 'Fehler beim Laden der Patienten', 'error');
                }
            } catch (error) {
                console.error('Error:', error);
                showAlert('Verbindungsfehler: ' + error.message, 'error');
            }
        }

        function renderPatientsTable(patients) {
            if (patients.length === 0) {
                document.getElementById('patientsTable').innerHTML = '<p style="text-align: center; color: var(--gray-500); padding: 2rem;">Keine Patienten gefunden</p>';
                return;
            }

            const html = `
                <table>
                    <thead>
                        <tr>
                            <th>Name</th>
                            <th>Zimmer</th>
                            <th>Abteilung</th>
                            <th>Arzt</th>
                            <th>Aufnahmedatum</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${patients.map(p => `
                            <tr>
                                <td><strong>${e(p.first_name)} ${e(p.last_name)}</strong></td>
                                <td>${e(p.room_number)}</td>
                                <td>${e(p.department)}</td>
                                <td>${e(p.doctor)}</td>
                                <td>${formatDate(p.admission_date)}</td>
                                <td><span class="badge ${p.status === 'aufgenommen' ? 'badge-success' : 'badge-warning'}">${p.status === 'aufgenommen' ? 'Aufgenommen' : 'Entlassen'}</span></td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>
            `;
            document.getElementById('patientsTable').innerHTML = html;
        }

        // Form submission
        document.getElementById('patientForm').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            const data = Object.fromEntries(formData.entries());
            
            // Handle checkbox
            data.notifyEmergencyContact = document.getElementById('notifyToggle').checked ? 'true' : 'false';
            
            try {
                const response = await fetch('../api/patients.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-Token': csrfToken
                    },
                    body: JSON.stringify(data)
                });

                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                }

                const result = await response.json();
                
                if (result.success) {
                    let message = 'Patient erfolgreich aufgenommen';
                    
                    // Check if notification was sent
                    if (result.data.notification) {
                        const notif = result.data.notification;
                        if (notif.email_sent && notif.sms_sent) {
                            message += '. E-Mail und SMS wurden gesendet.';
                        } else if (notif.email_sent) {
                            message += '. E-Mail wurde gesendet.';
                        } else if (notif.sms_sent) {
                            message += '. SMS wurde gesendet.';
                        }
                        
                        if (!notif.success) {
                            message += ' (Hinweis: ' + notif.message + ')';
                        }
                    }
                    
                    showAlert(message, 'success');
                    closeModal();
                    this.reset();
                    loadPatients();
                } else {
                    showAlert(result.error || 'Fehler beim Erstellen des Patienten', 'error');
                }
            } catch (error) {
                console.error('Error:', error);
                showAlert('Verbindungsfehler: ' + error.message, 'error');
            }
        });

        function openModal() {
            document.getElementById('patientModal').classList.add('show');
            document.body.style.overflow = 'hidden';
        }

        function closeModal() {
            document.getElementById('patientModal').classList.remove('show');
            document.body.style.overflow = '';
        }

        function showAlert(message, type = 'info') {
            const container = document.getElementById('alertContainer');
            const alert = document.createElement('div');
            alert.className = `alert alert-${type}`;
            
            const icons = {
                success: '<svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/></svg>',
                error: '<svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/></svg>',
                warning: '<svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"/></svg>'
            };
            
            alert.innerHTML = `${icons[type] || icons.info}<span>${message}</span>`;
            container.appendChild(alert);
            
            setTimeout(() => alert.remove(), 5000);
        }

        function debounce(func, wait) {
            let timeout;
            return function executedFunction(...args) {
                const later = () => {
                    clearTimeout(timeout);
                    func(...args);
                };
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
            };
        }

        function formatDate(dateString) {
            if (!dateString) return '-';
            return new Date(dateString).toLocaleDateString('de-DE');
        }

        function e(string) {
            if (!string) return '';
            return string.replace(/[&<>"']/g, function(m) {
                return {'&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#39;'}[m];
            });
        }

        // Close modal on outside click
        document.getElementById('patientModal').addEventListener('click', function(e) {
            if (e.target === this) closeModal();
        });
    </script>
</body>
</html>