<?php
/**
 * Ars Medical Frankfurt - Admin Login Page
 */

if (!defined('SECURE_ACCESS')) {
    define('SECURE_ACCESS', true);
}

require_once __DIR__ . '/../includes/functions.php';

// Redirect if already logged in
if (isLoggedIn()) {
    header('Location: /admin/index.php');
    exit;
}

$error = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // This would normally be handled by AJAX, but we'll process it here as fallback
    $username = sanitizeInput($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';
    $csrfToken = $_POST['csrf_token'] ?? '';
    
    if (!validateCSRFToken($csrfToken)) {
        $error = 'Invalid security token. Please refresh the page and try again.';
    } elseif (empty($username) || empty($password)) {
        $error = 'Please enter both username and password.';
    }
}
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Login | <?php echo e(getSetting('hospital_name', 'Ars Medical Frankfurt')); ?></title>
    <meta name="description" content="Secure admin login for <?php echo e(getSetting('hospital_name', 'Ars Medical Frankfurt')); ?> hospital management system.">
    <?php
    $faviconUrl = getSetting('logo_favicon', '');
    if ($faviconUrl):
    ?>
        <link rel="icon" type="image/png" href="<?php echo e($faviconUrl); ?>">
    <?php endif; ?>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary: #004B7C;
            --primary-dark: #003a61;
            --primary-light: #0066a8;
            --secondary: #00A896;
            --danger: #DC2626;
            --success: #10B981;
            --white: #FFFFFF;
            --gray-50: #F8FAFC;
            --gray-100: #F1F5F9;
            --gray-200: #E2E8F0;
            --gray-300: #CBD5E1;
            --gray-400: #94A3B8;
            --gray-500: #64748B;
            --gray-600: #475569;
            --gray-700: #334155;
            --gray-800: #1E293B;
            --gray-900: #0F172A;
        }

        * { margin: 0; padding: 0; box-sizing: border-box; }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 1rem;
        }

        .login-container {
            width: 100%;
            max-width: 420px;
        }

        .login-box {
            background: var(--white);
            border-radius: 16px;
            box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
            padding: 2.5rem;
        }

        .login-header {
            text-align: center;
            margin-bottom: 2rem;
        }

        .logo {
            display: inline-flex;
            align-items: center;
            gap: 0.75rem;
            text-decoration: none;
            margin-bottom: 1.5rem;
        }

        .logo-icon {
            width: 50px;
            height: 50px;
            background: var(--primary);
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .logo-icon svg {
            width: 28px;
            height: 28px;
            color: var(--white);
        }

        .logo-text h1 {
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--gray-900);
            line-height: 1.2;
        }

        .logo-text span {
            font-size: 0.75rem;
            color: var(--gray-500);
            text-transform: uppercase;
            letter-spacing: 0.1em;
        }

        .login-header p {
            color: var(--gray-500);
            font-size: 0.9375rem;
        }

        .alert {
            padding: 0.875rem 1rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
            font-size: 0.875rem;
            display: none;
        }

        .alert.show {
            display: block;
        }

        .alert-error {
            background: #FEE2E2;
            border: 1px solid #FECACA;
            color: var(--danger);
        }

        .alert-success {
            background: #D1FAE5;
            border: 1px solid #A7F3D0;
            color: var(--success);
        }

        .form-group {
            margin-bottom: 1.25rem;
        }

        .form-group label {
            display: block;
            font-weight: 500;
            color: var(--gray-700);
            margin-bottom: 0.5rem;
            font-size: 0.875rem;
        }

        .form-group input {
            width: 100%;
            padding: 0.75rem 1rem;
            border: 1px solid var(--gray-300);
            border-radius: 8px;
            font-size: 1rem;
            font-family: inherit;
            transition: border-color 0.2s, box-shadow 0.2s;
        }

        .form-group input:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(0, 75, 124, 0.1);
        }

        .form-group input::placeholder {
            color: var(--gray-400);
        }

        .btn {
            width: 100%;
            padding: 0.875rem 1.5rem;
            background: var(--primary);
            color: var(--white);
            border: none;
            border-radius: 8px;
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            transition: background 0.2s, transform 0.1s;
        }

        .btn:hover {
            background: var(--primary-dark);
        }

        .btn:active {
            transform: scale(0.98);
        }

        .btn:disabled {
            background: var(--gray-400);
            cursor: not-allowed;
        }

        .login-footer {
            text-align: center;
            margin-top: 1.5rem;
            padding-top: 1.5rem;
            border-top: 1px solid var(--gray-200);
        }

        .login-footer a {
            color: var(--primary);
            text-decoration: none;
            font-size: 0.875rem;
        }

        .login-footer a:hover {
            text-decoration: underline;
        }

        .back-link {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            color: var(--white);
            text-decoration: none;
            font-size: 0.875rem;
            margin-top: 1.5rem;
            opacity: 0.8;
            transition: opacity 0.2s;
        }

        .back-link:hover {
            opacity: 1;
        }

        .spinner {
            display: none;
            width: 20px;
            height: 20px;
            border: 2px solid var(--white);
            border-top-color: transparent;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin-right: 0.5rem;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }

        .loading .spinner {
            display: inline-block;
        }

        .loading .btn-text {
            display: inline-flex;
            align-items: center;
        }
    </style>
</head>
<body>
    <div class="login-container">
        <div class="login-box">
            <div class="login-header">
                <?php
                $logoUrl = getSetting('logo_url', '');
                if ($logoUrl):
                ?>
                    <a href="../index.php" style="display: flex; align-items: center; justify-content: center; margin-bottom: 1rem;">
                        <img src="<?php echo e($logoUrl); ?>" alt="<?php echo e(getSetting('hospital_name', 'Ars Medical Frankfurt')); ?>" style="max-height: 60px; max-width: 200px; width: auto;">
                    </a>
                <?php else: ?>
                    <a href="../index.php" class="logo">
                        <div class="logo-icon">
                            <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"/></svg>
                        </div>
                        <div class="logo-text">
                            <h1>Ars Medical</h1>
                            <span>Admin Login</span>
                        </div>
                    </a>
                <?php endif; ?>
                <p>Internes Verwaltungssystem</p>
            </div>

            <div class="alert alert-error" id="errorAlert">
                <?php echo $error; ?>
            </div>

            <div class="alert alert-success" id="successAlert"></div>

            <?php
            // Generate or get CSRF token
            if (empty($_SESSION['csrf_token'])) {
                $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
            }
            $csrfToken = $_SESSION['csrf_token'];
            ?>
            <form id="loginForm" method="POST" action="">
                <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrfToken); ?>">
                
                <div class="form-group">
                    <label for="username">Benutzername</label>
                    <input type="text" id="username" name="username" required 
                           placeholder="admin" autocomplete="username">
                </div>

                <div class="form-group">
                    <label for="password">Passwort</label>
                    <input type="password" id="password" name="password" required 
                           placeholder="••••••••" autocomplete="current-password">
                </div>

                <button type="submit" class="btn" id="loginBtn">
                    <span class="btn-text">
                        <span class="spinner"></span>
                        Anmelden
                    </span>
                </button>
            </form>

            <div class="login-footer">
                <a href="../index.php">Zurück zur Website</a>
            </div>
        </div>

        <a href="../index.php" class="back-link">
            <svg width="16" height="16" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 19l-7-7m0 0l7-7m-7 7h18"/></svg>
            Zurück zur Startseite
        </a>
    </div>

    <script>
        document.getElementById('loginForm').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const btn = document.getElementById('loginBtn');
            const errorAlert = document.getElementById('errorAlert');
            const successAlert = document.getElementById('successAlert');
            
            // Hide alerts
            errorAlert.classList.remove('show');
            successAlert.classList.remove('show');
            
            // Show loading
            btn.classList.add('loading');
            btn.disabled = true;
            
            const formData = new FormData(this);
            formData.append('action', 'login');
            
            try {
                const response = await fetch('../api/auth.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await response.json();
                
                if (data.success) {
                    successAlert.textContent = 'Anmeldung erfolgreich. Weiterleitung...';
                    successAlert.classList.add('show');
                    
                    // Redirect to admin dashboard
                    setTimeout(() => {
                        window.location.href = 'index.php';
                    }, 500);
                } else {
                    errorAlert.textContent = data.error || 'Anmeldung fehlgeschlagen.';
                    errorAlert.classList.add('show');
                }
            } catch (error) {
                errorAlert.textContent = 'Ein Fehler ist aufgetreten. Bitte versuchen Sie es erneut.';
                errorAlert.classList.add('show');
            } finally {
                btn.classList.remove('loading');
                btn.disabled = false;
            }
        });

        // Show PHP error if present
        <?php if (!empty($error)): ?>
        document.getElementById('errorAlert').classList.add('show');
        <?php endif; ?>
    </script>
</body>
</html>