<?php
/**
 * Ars Medical Frankfurt - Invoice Management
 * Admin page for managing invoices
 */

if (!defined('SECURE_ACCESS')) {
    define('SECURE_ACCESS', true);
}

require_once __DIR__ . '/../includes/functions.php';

// Require authentication
requireAuth();

// Get current user
$currentUser = getCurrentUser();

// Get all patients for dropdown
try {
    $db = getDB();
    $patients = $db->fetchAll("SELECT id, first_name, last_name FROM patients WHERE status = 'aufgenommen' ORDER BY last_name, first_name");
} catch (Exception $e) {
    $patients = [];
}
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Rechnungsverwaltung | Ars Medical Frankfurt</title>
    <meta name="description" content="Invoice management for Ars Medical Frankfurt hospital.">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary: #004B7C; --primary-dark: #003a61; --primary-light: #0066a8;
            --secondary: #00A896; --accent: #E8F4F8; --white: #FFFFFF;
            --gray-50: #F8FAFC; --gray-100: #F1F5F9; --gray-200: #E2E8F0;
            --gray-300: #CBD5E1; --gray-400: #94A3B8; --gray-500: #64748B;
            --gray-600: #475569; --gray-700: #334155; --gray-800: #1E293B; --gray-900: #0F172A;
            --danger: #DC2626; --danger-light: #FEE2E2; --warning: #F59E0B;
            --warning-light: #FEF3C7; --success: #10B981; --success-light: #D1FAE5;
            --info: #3B82F6; --info-light: #DBEAFE;
            --shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
            --shadow: 0 1px 3px 0 rgb(0 0 0 / 0.1), 0 1px 2px -1px rgb(0 0 0 / 0.1);
            --shadow-md: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
            --shadow-lg: 0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1);
        }

        * { margin: 0; padding: 0; box-sizing: border-box; }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: var(--gray-100);
            color: var(--gray-800);
            line-height: 1.6;
        }

        .admin-container { display: flex; min-height: 100vh; }

        /* Sidebar */
        .sidebar {
            width: 260px;
            background: var(--white);
            border-right: 1px solid var(--gray-200);
            display: flex;
            flex-direction: column;
            position: fixed;
            height: 100vh;
            z-index: 100;
        }

        .sidebar-header {
            padding: 1.5rem;
            border-bottom: 1px solid var(--gray-200);
        }

        .sidebar-header a {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            text-decoration: none;
        }

        .sidebar-header .logo-icon {
            width: 40px;
            height: 40px;
            background: var(--primary);
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .sidebar-header .logo-icon svg { width: 24px; height: 24px; color: var(--white); }

        .sidebar-header .logo-text h1 {
            font-size: 1.125rem;
            font-weight: 700;
            color: var(--gray-900);
            line-height: 1.2;
        }

        .sidebar-header .logo-text span {
            font-size: 0.6875rem;
            color: var(--gray-500);
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }

        .sidebar-nav {
            flex: 1;
            padding: 1rem 0.75rem;
            overflow-y: auto;
        }

        .sidebar-section { margin-bottom: 1.5rem; }

        .sidebar-section-title {
            font-size: 0.6875rem;
            font-weight: 600;
            color: var(--gray-400);
            text-transform: uppercase;
            letter-spacing: 0.1em;
            padding: 0 0.75rem;
            margin-bottom: 0.5rem;
        }

        .sidebar-nav a {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem;
            color: var(--gray-600);
            text-decoration: none;
            font-size: 0.9375rem;
            font-weight: 500;
            border-radius: 8px;
            transition: all 0.2s;
            margin-bottom: 0.25rem;
        }

        .sidebar-nav a:hover { background: var(--gray-100); color: var(--gray-900); }
        .sidebar-nav a.active { background: var(--accent); color: var(--primary); }
        .sidebar-nav a svg { width: 20px; height: 20px; }

        .sidebar-footer {
            padding: 1rem 0.75rem;
            border-top: 1px solid var(--gray-200);
        }

        .sidebar-footer .user-menu {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem;
            cursor: pointer;
            border-radius: 8px;
            transition: background 0.2s;
        }

        .sidebar-footer .user-menu:hover { background: var(--gray-100); }

        .user-avatar {
            width: 36px;
            height: 36px;
            background: var(--primary);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: var(--white);
            font-weight: 600;
            font-size: 0.875rem;
        }

        .user-info { flex: 1; }
        .user-info strong { display: block; font-size: 0.875rem; color: var(--gray-900); }
        .user-info span { font-size: 0.75rem; color: var(--gray-500); }

        .logout-btn {
            background: none;
            border: none;
            cursor: pointer;
            padding: 0.5rem;
            color: var(--gray-400);
            transition: color 0.2s;
        }

        .logout-btn:hover { color: var(--danger); }

        /* Main Content */
        .main-content {
            flex: 1;
            margin-left: 260px;
            padding: 2rem;
        }

        /* Header */
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
        }

        .page-header h1 {
            font-size: 1.875rem;
            font-weight: 700;
            color: var(--gray-900);
        }

        .user-menu {
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .user-menu span { color: var(--gray-600); }

        .btn {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.625rem 1.25rem;
            border-radius: 6px;
            font-size: 0.875rem;
            font-weight: 500;
            text-decoration: none;
            cursor: pointer;
            transition: all 0.2s;
            border: none;
        }

        .btn-primary { background: var(--primary); color: var(--white); }
        .btn-primary:hover { background: var(--primary-dark); }
        .btn-secondary { background: var(--gray-200); color: var(--gray-700); }
        .btn-secondary:hover { background: var(--gray-300); }
        .btn-success { background: var(--success); color: var(--white); }
        .btn-success:hover { background: #059669; }

        /* Stats Grid */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(240px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .stat-card {
            background: var(--white);
            padding: 1.5rem;
            border-radius: 12px;
            box-shadow: var(--shadow);
        }

        .stat-card-header {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            margin-bottom: 1rem;
        }

        .stat-card-icon {
            width: 48px;
            height: 48px;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .stat-card-icon svg { width: 24px; height: 24px; }

        .stat-card-icon.primary { background: var(--primary-light); color: var(--white); }
        .stat-card-icon.success { background: var(--success); color: var(--white); }
        .stat-card-icon.warning { background: var(--warning); color: var(--white); }
        .stat-card-icon.danger { background: var(--danger); color: var(--white); }

        .stat-card-title { font-size: 0.875rem; color: var(--gray-500); }
        .stat-card-value { font-size: 1.875rem; font-weight: 700; color: var(--gray-900); }

        /* Content Card */
        .content-card {
            background: var(--white);
            border-radius: 12px;
            box-shadow: var(--shadow);
            overflow: hidden;
        }

        .content-card-header {
            padding: 1.5rem;
            border-bottom: 1px solid var(--gray-200);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .content-card-title { font-size: 1.25rem; font-weight: 600; }

        .content-card-body { padding: 1.5rem; }

        /* Table */
        .table-container { overflow-x: auto; }

        table {
            width: 100%;
            border-collapse: collapse;
        }

        th, td {
            padding: 1rem;
            text-align: left;
            border-bottom: 1px solid var(--gray-200);
        }

        th {
            font-weight: 600;
            color: var(--gray-700);
            background: var(--gray-50);
            font-size: 0.875rem;
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }

        td { font-size: 0.9375rem; }

        tr:hover { background: var(--gray-50); }

        .badge {
            display: inline-flex;
            align-items: center;
            padding: 0.25rem 0.75rem;
            border-radius: 9999px;
            font-size: 0.75rem;
            font-weight: 500;
        }

        .badge-success { background: var(--success-light); color: #065F46; }
        .badge-warning { background: var(--warning-light); color: #92400E; }
        .badge-danger { background: var(--danger-light); color: #991B1B; }
        .badge-info { background: var(--info-light); color: #1E40AF; }

        .btn-sm { padding: 0.375rem 0.75rem; font-size: 0.8125rem; }

        /* Modal */
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            align-items: center;
            justify-content: center;
            padding: 1rem;
        }

        .modal.active { display: flex; }

        .modal-content {
            background: var(--white);
            border-radius: 12px;
            max-width: 600px;
            width: 100%;
            max-height: 90vh;
            overflow-y: auto;
            box-shadow: var(--shadow-lg);
        }

        .modal-header {
            padding: 1.5rem;
            border-bottom: 1px solid var(--gray-200);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .modal-title { font-size: 1.25rem; font-weight: 600; }

        .modal-close {
            background: none;
            border: none;
            cursor: pointer;
            color: var(--gray-400);
            padding: 0.5rem;
        }

        .modal-close:hover { color: var(--gray-600); }

        .modal-body { padding: 1.5rem; }

        .modal-footer {
            padding: 1.5rem;
            border-top: 1px solid var(--gray-200);
            display: flex;
            justify-content: flex-end;
            gap: 0.75rem;
        }

        /* Form */
        .form-group { margin-bottom: 1.25rem; }

        .form-group label {
            display: block;
            font-weight: 500;
            color: var(--gray-700);
            margin-bottom: 0.5rem;
            font-size: 0.875rem;
        }

        .form-control {
            width: 100%;
            padding: 0.625rem 0.875rem;
            border: 1px solid var(--gray-300);
            border-radius: 6px;
            font-size: 0.9375rem;
            font-family: inherit;
            transition: border-color 0.2s, box-shadow 0.2s;
        }

        .form-control:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(0, 75, 124, 0.1);
        }

        .form-row {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 1rem;
        }

        /* Alert */
        .alert {
            padding: 1rem 1.25rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
            display: none;
        }

        .alert.show { display: block; }

        .alert-success { background: var(--success-light); color: #065F46; border: 1px solid #A7F3D0; }
        .alert-error { background: var(--danger-light); color: #991B1B; border: 1px solid #FECACA; }

        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 3rem 1.5rem;
            color: var(--gray-500);
        }

        .empty-state svg {
            width: 64px;
            height: 64px;
            margin-bottom: 1rem;
            opacity: 0.5;
        }

        @media (max-width: 1024px) {
            .sidebar { transform: translateX(-100%); transition: transform 0.3s; }
            .sidebar.active { transform: translateX(0); }
            .main-content { margin-left: 0; }
            .form-row { grid-template-columns: 1fr; }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Sidebar -->
        <aside class="sidebar">
            <div class="sidebar-header">
                <a href="index.php">
                    <div class="logo-icon">
                        <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"/>
                        </svg>
                    </div>
                    <div class="logo-text">
                        <h1>Ars Medical</h1>
                        <span>Admin Dashboard</span>
                    </div>
                </a>
            </div>
            <nav class="sidebar-nav">
                <div class="sidebar-section">
                    <div class="sidebar-section-title">Hauptmenü</div>
                    <a href="index.php">
                        <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2H6a2 2 0 01-2-2V6zM14 6a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2h-2a2 2 0 01-2-2V6zM4 16a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2H6a2 2 0 01-2-2v-2zM14 16a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2h-2a2 2 0 01-2-2v-2z"/></svg>
                        Dashboard
                    </a>
                    <a href="patients.php">
                        <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z"/></svg>
                        Patienten
                    </a>
                    <a href="invoices.php" class="active">
                        <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/></svg>
                        Rechnungen
                    </a>
                </div>
                <div class="sidebar-section">
                    <div class="sidebar-section-title">System</div>
                    <a href="settings.php">
                        <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z"/><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/></svg>
                        Einstellungen
                    </a>
                    <a href="logout.php">
                        <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"/></svg>
                        Abmelden
                    </a>
                </div>
            </nav>
            <div class="sidebar-footer">
                <div class="user-menu">
                    <div class="user-avatar"><?php echo substr($_SESSION['full_name'] ?? 'AD', 0, 2); ?></div>
                    <div class="user-info">
                        <strong><?php echo e($_SESSION['full_name'] ?? 'Admin'); ?></strong>
                        <span><?php echo e($_SESSION['email'] ?? 'admin@arsmedica.de'); ?></span>
                    </div>
                    <a href="logout.php" class="logout-btn" title="Abmelden">
                        <svg width="20" height="20" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"/></svg>
                    </a>
                </div>
            </div>
        </aside>

        <!-- Main Content -->
        <main class="main-content">
            <div class="page-header">
                <h1>Rechnungsverwaltung</h1>
            </div>

            <div id="alertSuccess" class="alert alert-success"></div>
            <div id="alertError" class="alert alert-error"></div>

            <!-- Stats -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-card-header">
                        <div class="stat-card-icon primary">
                            <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/></svg>
                        </div>
                        <div class="stat-card-title">Gesamtrechnungen</div>
                    </div>
                    <div class="stat-card-value" id="totalInvoices">-</div>
                </div>
                <div class="stat-card">
                    <div class="stat-card-header">
                        <div class="stat-card-icon warning">
                            <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/></svg>
                        </div>
                        <div class="stat-card-title">Offen</div>
                    </div>
                    <div class="stat-card-value" id="pendingInvoices">-</div>
                </div>
                <div class="stat-card">
                    <div class="stat-card-header">
                        <div class="stat-card-icon success">
                            <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/></svg>
                        </div>
                        <div class="stat-card-title">Bezahlt</div>
                    </div>
                    <div class="stat-card-value" id="paidInvoices">-</div>
                </div>
                <div class="stat-card">
                    <div class="stat-card-header">
                        <div class="stat-card-icon danger">
                            <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/></svg>
                        </div>
                        <div class="stat-card-title">Gesamtbetrag</div>
                    </div>
                    <div class="stat-card-value" id="totalAmount">-</div>
                </div>
            </div>

            <!-- Invoices List -->
            <div class="content-card">
                <div class="content-card-header">
                    <h2 class="content-card-title">Alle Rechnungen</h2>
                    <button class="btn btn-primary" onclick="openCreateModal()">
                        <svg width="16" height="16" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"/></svg>
                        Neue Rechnung
                    </button>
                </div>
                <div class="content-card-body">
                    <div class="table-container">
                        <table id="invoicesTable">
                            <thead>
                                <tr>
                                    <th>Rechnungsnr.</th>
                                    <th>Patient</th>
                                    <th>Datum</th>
                                    <th>Betrag</th>
                                    <th>Status</th>
                                    <th>Aktionen</th>
                                </tr>
                            </thead>
                            <tbody id="invoicesTableBody">
                                <tr>
                                    <td colspan="6" class="empty-state">
                                        <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/></svg>
                                        <p>Rechnungen werden geladen...</p>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <!-- Create Invoice Modal -->
    <div class="modal" id="createModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">Neue Rechnung erstellen</h3>
                <button class="modal-close" onclick="closeCreateModal()">
                    <svg width="24" height="24" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/></svg>
                </button>
            </div>
            <div class="modal-body">
                <form id="createInvoiceForm">
                    <div class="form-group">
                        <label for="patientId">Patient</label>
                        <select class="form-control" id="patientId" name="patient_id" required>
                            <option value="">Patient auswählen...</option>
                            <?php foreach ($patients as $patient): ?>
                                <option value="<?php echo $patient['id']; ?>">
                                    <?php echo e($patient['last_name'] . ', ' . $patient['first_name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-row">
                        <div class="form-group">
                            <label for="treatmentCost">Behandlungskosten (€)</label>
                            <input type="number" class="form-control" id="treatmentCost" name="treatment_cost" step="0.01" min="0" value="0">
                        </div>
                        <div class="form-group">
                            <label for="medicationCost">Medikamentenkosten (€)</label>
                            <input type="number" class="form-control" id="medicationCost" name="medication_cost" step="0.01" min="0" value="0">
                        </div>
                    </div>
                    <div class="form-row">
                        <div class="form-group">
                            <label for="roomCost">Zimmerkosten pro Tag (€)</label>
                            <input type="number" class="form-control" id="roomCost" name="room_cost_per_day" step="0.01" min="0" value="150">
                        </div>
                        <div class="form-group">
                            <label for="stayDays">Aufenthaltstage</label>
                            <input type="number" class="form-control" id="stayDays" name="stay_days" min="1" value="1">
                        </div>
                    </div>
                    <div class="form-group">
                        <label for="doctorFee">Arzthonorar (€)</label>
                        <input type="number" class="form-control" id="doctorFee" name="doctor_fee" step="0.01" min="0" value="0">
                    </div>
                    <div class="form-group">
                        <label for="notes">Notizen</label>
                        <textarea class="form-control" id="notes" name="notes" rows="3"></textarea>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button class="btn btn-secondary" onclick="closeCreateModal()">Abbrechen</button>
                <button class="btn btn-primary" onclick="createInvoice()">Rechnung erstellen</button>
            </div>
        </div>
    </div>

    <script>
        // CSRF Token from PHP session
        const csrfToken = '<?php echo generateCSRFToken(); ?>';

        // Load invoices on page load
        document.addEventListener('DOMContentLoaded', loadInvoices);

        async function loadInvoices() {
            try {
                const response = await fetch('../api/invoices.php', {
                    headers: {
                        'X-CSRF-Token': csrfToken
                    }
                });

                const responseText = await response.text();
                let data;
                try {
                    data = JSON.parse(responseText);
                } catch (e) {
                    console.error('Non-JSON response:', responseText);
                    throw new Error(`Server error: ${response.status}. Check server logs.`);
                }

                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}: ${data.error || response.statusText}`);
                }

                if (data.success) {
                    const invoices = data.data || [];
                    displayInvoices(invoices);
                    updateStats(invoices);
                } else {
                    showAlert('error', data.error || 'Fehler beim Laden der Rechnungen');
                }
            } catch (error) {
                showAlert('error', 'Verbindungsfehler: ' + error.message);
            }
        }

        function displayInvoices(invoices) {
            const tbody = document.getElementById('invoicesTableBody');

            if (!invoices || invoices.length === 0) {
                tbody.innerHTML = `
                    <tr>
                        <td colspan="6" class="empty-state">
                            <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/></svg>
                            <p>Keine Rechnungen vorhanden</p>
                        </td>
                    </tr>
                `;
                return;
            }

            tbody.innerHTML = invoices.map(invoice => {
                const statusBadge = getStatusBadge(invoice.status);
                return `
                    <tr>
                        <td><strong>${invoice.invoice_number}</strong></td>
                        <td>${e(invoice.patient_name)}</td>
                        <td>${formatDate(invoice.invoice_date)}</td>
                        <td>${formatCurrency(invoice.total_amount)}</td>
                        <td>${statusBadge}</td>
                        <td>
                            <button class="btn btn-sm btn-secondary" onclick="viewInvoice(${invoice.id})">Ansehen</button>
                            ${invoice.status === 'offen' ? `<button class="btn btn-sm btn-success" onclick="markPaid(${invoice.id})">Bezahlt</button>` : ''}
                        </td>
                    </tr>
                `;
            }).join('');
        }

        function updateStats(invoices) {
            const total = invoices.length;
            const pending = invoices.filter(i => i.status === 'offen').length;
            const paid = invoices.filter(i => i.status === 'bezahlt').length;
            const totalAmount = invoices.reduce((sum, i) => sum + parseFloat(i.total_amount), 0);

            document.getElementById('totalInvoices').textContent = total;
            document.getElementById('pendingInvoices').textContent = pending;
            document.getElementById('paidInvoices').textContent = paid;
            document.getElementById('totalAmount').textContent = formatCurrency(totalAmount);
        }

        function getStatusBadge(status) {
            const badges = {
                'offen': '<span class="badge badge-warning">Offen</span>',
                'bezahlt': '<span class="badge badge-success">Bezahlt</span>',
                'ueberfaellig': '<span class="badge badge-danger">Überfällig</span>',
                'storniert': '<span class="badge badge-info">Storniert</span>'
            };
            return badges[status] || `<span class="badge">${status}</span>`;
        }

        function formatDate(dateString) {
            if (!dateString) return '-';
            const date = new Date(dateString);
            return date.toLocaleDateString('de-DE');
        }

        function formatCurrency(amount) {
            return new Intl.NumberFormat('de-DE', { style: 'currency', currency: 'EUR' }).format(amount);
        }

        function e(text) {
            if (!text) return '';
            return text.replace(/[&<>"']/g, m => ({ '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#39;' })[m]);
        }

        function openCreateModal() {
            document.getElementById('createModal').classList.add('active');
        }

        function closeCreateModal() {
            document.getElementById('createModal').classList.remove('active');
            document.getElementById('createInvoiceForm').reset();
        }

        async function createInvoice() {
            const form = document.getElementById('createInvoiceForm');
            const formData = new FormData(form);

            if (!formData.get('patient_id')) {
                showAlert('error', 'Bitte wählen Sie einen Patienten aus');
                return;
            }

            // Add CSRF token to form data
            formData.append('csrf_token', csrfToken);

            try {
                const response = await fetch('../api/invoices.php', {
                    method: 'POST',
                    body: formData
                });

                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                }

                const data = await response.json();

                if (data.success) {
                    showAlert('success', 'Rechnung erfolgreich erstellt!');
                    closeCreateModal();
                    loadInvoices();
                } else {
                    showAlert('error', data.error || 'Fehler beim Erstellen der Rechnung');
                }
            } catch (error) {
                showAlert('error', 'Verbindungsfehler: ' + error.message);
            }
        }

        async function markPaid(invoiceId) {
            if (!confirm('Rechnung als bezahlt markieren?')) return;

            try {
                const response = await fetch(`../api/invoices.php?id=${invoiceId}&action=mark_paid`, {
                    method: 'PUT',
                    headers: {
                        'X-CSRF-Token': csrfToken
                    }
                });

                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                }

                const data = await response.json();

                if (data.success) {
                    showAlert('success', 'Rechnung als bezahlt markiert!');
                    loadInvoices();
                } else {
                    showAlert('error', data.error || 'Fehler beim Aktualisieren');
                }
            } catch (error) {
                showAlert('error', 'Verbindungsfehler: ' + error.message);
            }
        }

        function viewInvoice(invoiceId) {
            window.open(`../api/invoices.php?id=${invoiceId}&format=pdf`, '_blank');
        }

        function showAlert(type, message) {
            const alertEl = document.getElementById(type === 'success' ? 'alertSuccess' : 'alertError');
            alertEl.textContent = message;
            alertEl.classList.add('show');

            setTimeout(() => {
                alertEl.classList.remove('show');
            }, 5000);
        }

        // Close modal on outside click
        document.getElementById('createModal').addEventListener('click', function(e) {
            if (e.target === this) closeCreateModal();
        });
    </script>
</body>
</html>
