<?php
/**
 * Ars Medical Frankfurt - Admin Dashboard
 * Secure admin panel with MySQL backend
 */

if (!defined('SECURE_ACCESS')) {
    define('SECURE_ACCESS', true);
}

require_once __DIR__ . '/../includes/functions.php';

// Require authentication
requireAuth();

// Get current user
$currentUser = getCurrentUser();
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard | <?php echo e(getSetting('hospital_name', 'Ars Medical Frankfurt')); ?></title>
    <meta name="description" content="Secure admin dashboard for <?php echo e(getSetting('hospital_name', 'Ars Medical Frankfurt')); ?> hospital management.">
    <?php
    $faviconUrl = getSetting('logo_favicon', '');
    if ($faviconUrl):
    ?>
        <link rel="icon" type="image/png" href="<?php echo e($faviconUrl); ?>">
    <?php endif; ?>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary: #004B7C;
            --primary-dark: #003a61;
            --primary-light: #0066a8;
            --secondary: #00A896;
            --accent: #E8F4F8;
            --white: #FFFFFF;
            --gray-50: #F8FAFC;
            --gray-100: #F1F5F9;
            --gray-200: #E2E8F0;
            --gray-300: #CBD5E1;
            --gray-400: #94A3B8;
            --gray-500: #64748B;
            --gray-600: #475569;
            --gray-700: #334155;
            --gray-800: #1E293B;
            --gray-900: #0F172A;
            --danger: #DC2626;
            --danger-light: #FEE2E2;
            --warning: #F59E0B;
            --warning-light: #FEF3C7;
            --success: #10B981;
            --success-light: #D1FAE5;
            --info: #3B82F6;
            --info-light: #DBEAFE;
            --shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
            --shadow: 0 1px 3px 0 rgb(0 0 0 / 0.1), 0 1px 2px -1px rgb(0 0 0 / 0.1);
            --shadow-md: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
            --shadow-lg: 0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1);
        }

        * { margin: 0; padding: 0; box-sizing: border-box; }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            line-height: 1.6;
            color: var(--gray-800);
            background-color: var(--gray-100);
        }

        /* Sidebar */
        .sidebar {
            width: 260px;
            background: var(--white);
            border-right: 1px solid var(--gray-200);
            display: flex;
            flex-direction: column;
            position: fixed;
            height: 100vh;
            z-index: 100;
        }

        .sidebar-header {
            padding: 1.5rem;
            border-bottom: 1px solid var(--gray-200);
        }

        .sidebar-logo {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            text-decoration: none;
        }

        .sidebar-logo-icon {
            width: 40px;
            height: 40px;
            background: var(--primary);
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .sidebar-logo-icon svg { width: 24px; height: 24px; color: var(--white); }

        .sidebar-logo-text h2 {
            font-size: 1.125rem;
            font-weight: 700;
            color: var(--gray-900);
            line-height: 1.2;
        }

        .sidebar-logo-text span {
            font-size: 0.6875rem;
            color: var(--gray-500);
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }

        .sidebar-nav {
            flex: 1;
            padding: 1rem 0.75rem;
            overflow-y: auto;
        }

        .sidebar-section { margin-bottom: 1.5rem; }

        .sidebar-section-title {
            font-size: 0.6875rem;
            font-weight: 600;
            color: var(--gray-400);
            text-transform: uppercase;
            letter-spacing: 0.1em;
            padding: 0 0.75rem;
            margin-bottom: 0.5rem;
        }

        .sidebar-nav a {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem;
            color: var(--gray-600);
            text-decoration: none;
            font-size: 0.9375rem;
            font-weight: 500;
            border-radius: 8px;
            transition: all 0.2s;
            margin-bottom: 0.25rem;
        }

        .sidebar-nav a:hover { background: var(--gray-100); color: var(--gray-900); }
        .sidebar-nav a.active { background: var(--accent); color: var(--primary); }
        .sidebar-nav a svg { width: 20px; height: 20px; }

        .sidebar-footer {
            padding: 1rem 0.75rem;
            border-top: 1px solid var(--gray-200);
        }

        .user-menu {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem;
            cursor: pointer;
            border-radius: 8px;
            transition: background 0.2s;
        }

        .user-menu:hover { background: var(--gray-100); }

        .user-avatar {
            width: 36px;
            height: 36px;
            background: var(--primary);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: var(--white);
            font-weight: 600;
            font-size: 0.875rem;
        }

        .user-info { flex: 1; }
        .user-info strong { display: block; font-size: 0.875rem; color: var(--gray-900); }
        .user-info span { font-size: 0.75rem; color: var(--gray-500); }

        .logout-btn {
            background: none;
            border: none;
            cursor: pointer;
            padding: 0.5rem;
            color: var(--gray-400);
            transition: color 0.2s;
        }

        .logout-btn:hover { color: var(--danger); }

        /* Main Content */
        .main-content {
            flex: 1;
            margin-left: 260px;
            padding: 2rem;
            min-height: 100vh;
        }

        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
        }

        .page-header h1 {
            font-size: 1.875rem;
            font-weight: 700;
            color: var(--gray-900);
        }

        /* Stats Grid */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(240px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .stat-card {
            background: var(--white);
            padding: 1.5rem;
            border-radius: 12px;
            box-shadow: var(--shadow-sm);
            border: 1px solid var(--gray-200);
        }

        .stat-card-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 1rem;
        }

        .stat-card-icon {
            width: 48px;
            height: 48px;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .stat-card-icon.blue { background: var(--info-light); color: var(--info); }
        .stat-card-icon.green { background: var(--success-light); color: var(--success); }
        .stat-card-icon.orange { background: var(--warning-light); color: var(--warning); }
        .stat-card-icon.red { background: var(--danger-light); color: var(--danger); }
        .stat-card-icon svg { width: 24px; height: 24px; }

        .stat-card-value {
            font-size: 2rem;
            font-weight: 700;
            color: var(--gray-900);
            margin-bottom: 0.25rem;
        }

        .stat-card-label {
            font-size: 0.875rem;
            color: var(--gray-500);
        }

        /* Cards */
        .card {
            background: var(--white);
            border-radius: 12px;
            box-shadow: var(--shadow-sm);
            border: 1px solid var(--gray-200);
            margin-bottom: 1.5rem;
        }

        .card-header {
            padding: 1.25rem 1.5rem;
            border-bottom: 1px solid var(--gray-200);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .card-header h2 {
            font-size: 1.125rem;
            font-weight: 600;
            color: var(--gray-900);
        }

        .card-body { padding: 1.5rem; }

        /* Tables */
        .table-container { overflow-x: auto; }

        table {
            width: 100%;
            border-collapse: collapse;
        }

        th, td {
            padding: 0.875rem 1rem;
            text-align: left;
            border-bottom: 1px solid var(--gray-200);
        }

        th {
            font-weight: 600;
            font-size: 0.75rem;
            text-transform: uppercase;
            letter-spacing: 0.05em;
            color: var(--gray-500);
            background: var(--gray-50);
        }

        td { font-size: 0.9375rem; color: var(--gray-700); }
        tr:hover td { background: var(--gray-50); }

        .badge {
            display: inline-flex;
            align-items: center;
            padding: 0.25rem 0.625rem;
            border-radius: 9999px;
            font-size: 0.75rem;
            font-weight: 600;
        }

        .badge-success { background: var(--success-light); color: var(--success); }
        .badge-warning { background: var(--warning-light); color: var(--warning); }
        .badge-danger { background: var(--danger-light); color: var(--danger); }
        .badge-info { background: var(--info-light); color: var(--info); }

        /* Buttons */
        .btn {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
            padding: 0.625rem 1.25rem;
            border-radius: 8px;
            font-weight: 600;
            font-size: 0.875rem;
            text-decoration: none;
            transition: all 0.2s;
            border: none;
            cursor: pointer;
        }

        .btn-primary { background: var(--primary); color: var(--white); }
        .btn-primary:hover { background: var(--primary-dark); }

        .btn-secondary {
            background: var(--white);
            color: var(--gray-700);
            border: 1px solid var(--gray-300);
        }

        .btn-secondary:hover { background: var(--gray-50); border-color: var(--gray-400); }
        .btn-sm { padding: 0.375rem 0.75rem; font-size: 0.8125rem; }

        /* Loading */
        .loading {
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 3rem;
            color: var(--gray-500);
        }

        .spinner {
            width: 24px;
            height: 24px;
            border: 2px solid var(--gray-300);
            border-top-color: var(--primary);
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin-right: 0.75rem;
        }

        @keyframes spin { to { transform: rotate(360deg); } }

        /* Responsive */
        @media (max-width: 1024px) {
            .sidebar { transform: translateX(-100%); transition: transform 0.3s; }
            .sidebar.show { transform: translateX(0); }
            .main-content { margin-left: 0; }
        }

        @media (max-width: 768px) {
            .stats-grid { grid-template-columns: 1fr; }
            .page-header { flex-direction: column; gap: 1rem; align-items: flex-start; }
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <aside class="sidebar">
        <div class="sidebar-header">
            <?php
            $logoUrl = getSetting('logo_url', '');
            if ($logoUrl):
            ?>
                <a href="../index.php" style="display: flex; align-items: center; gap: 0.75rem; text-decoration: none;">
                    <img src="<?php echo e($logoUrl); ?>" alt="<?php echo e(getSetting('hospital_name', 'Ars Medical Frankfurt')); ?>" style="max-height: 40px; max-width: 160px; width: auto;">
                </a>
            <?php else: ?>
                <a href="../index.php" class="sidebar-logo">
                    <div class="sidebar-logo-icon">
                        <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"/></svg>
                    </div>
                    <div class="sidebar-logo-text">
                        <h2>Ars Medical</h2>
                        <span>Admin Dashboard</span>
                    </div>
                </a>
            <?php endif; ?>
        </div>
        <nav class="sidebar-nav">
            <div class="sidebar-section">
                <div class="sidebar-section-title">Hauptmenü</div>
                <a href="index.php" class="active">
                    <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2H6a2 2 0 01-2-2V6zM14 6a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2h-2a2 2 0 01-2-2V6zM4 16a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2H6a2 2 0 01-2-2v-2zM14 16a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2h-2a2 2 0 01-2-2v-2z"/></svg>
                    Dashboard
                </a>
                <a href="patients.php">
                    <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z"/></svg>
                    Patienten
                </a>
                <a href="invoices.php">
                    <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/></svg>
                    Rechnungen
                </a>
            </div>
            <div class="sidebar-section">
                <div class="sidebar-section-title">System</div>
                <a href="settings.php">
                    <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z"/><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/></svg>
                    Einstellungen
                </a>
            </div>
        </nav>
        <div class="sidebar-footer">
            <div class="user-menu">
                <div class="user-avatar"><?php echo substr($currentUser['full_name'], 0, 2); ?></div>
                <div class="user-info">
                    <strong><?php echo e($currentUser['full_name']); ?></strong>
                    <span><?php echo e($currentUser['email']); ?></span>
                </div>
                <a href="logout.php" class="logout-btn" title="Abmelden">
                    <svg width="20" height="20" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"/></svg>
                </a>
            </div>
        </div>
    </aside>

    <!-- Main Content -->
    <main class="main-content">
        <div class="page-header">
            <h1>Dashboard</h1>
            <div style="display: flex; gap: 0.75rem;">
                <button class="btn btn-secondary btn-sm" onclick="refreshData()">
                    <svg width="16" height="16" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"/></svg>
                    Aktualisieren
                </button>
            </div>
        </div>

        <!-- Stats Grid -->
        <div class="stats-grid" id="statsGrid">
            <div class="loading">
                <div class="spinner"></div>
                Daten werden geladen...
            </div>
        </div>

        <!-- Recent Patients -->
        <div class="card">
            <div class="card-header">
                <h2>Neueste Patientenaufnahmen</h2>
                <a href="patients.php" class="btn btn-secondary btn-sm">Alle anzeigen</a>
            </div>
            <div class="card-body">
                <div class="table-container" id="recentPatientsTable">
                    <div class="loading">
                        <div class="spinner"></div>
                        Daten werden geladen...
                    </div>
                </div>
            </div>
        </div>

        <!-- Recent Invoices -->
        <div class="card">
            <div class="card-header">
                <h2>Neueste Rechnungen</h2>
                <a href="invoices.php" class="btn btn-secondary btn-sm">Alle anzeigen</a>
            </div>
            <div class="card-body">
                <div class="table-container" id="recentInvoicesTable">
                    <div class="loading">
                        <div class="spinner"></div>
                        Daten werden geladen...
                    </div>
                </div>
            </div>
        </div>
    </main>

    <script>
        // CSRF Token from PHP session
        const csrfToken = '<?php echo generateCSRFToken(); ?>';

        // Load dashboard data on page load
        document.addEventListener('DOMContentLoaded', loadDashboardData);

        async function loadDashboardData() {
            try {
                const response = await fetch('../api/dashboard.php', {
                    headers: {
                        'X-CSRF-Token': csrfToken
                    }
                });

                const responseText = await response.text();
                let data;
                try {
                    data = JSON.parse(responseText);
                } catch (e) {
                    // Response is not JSON - likely a PHP error
                    console.error('Non-JSON response:', responseText);
                    throw new Error(`Server error: ${response.status}. Check server logs.`);
                }

                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}: ${data.error || response.statusText}`);
                }

                if (data.success) {
                    renderStats(data.data.stats);
                    renderRecentPatients(data.data.recent_patients);
                    renderRecentInvoices(data.data.recent_invoices);
                } else {
                    showError(data.error || 'Fehler beim Laden der Daten');
                }
            } catch (error) {
                console.error('Error:', error);
                showError('Verbindungsfehler: ' + error.message);
            }
        }

        function renderStats(stats) {
            const html = `
                <div class="stat-card">
                    <div class="stat-card-header">
                        <div class="stat-card-icon blue">
                            <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z"/></svg>
                        </div>
                    </div>
                    <div class="stat-card-value">${stats.total_patients}</div>
                    <div class="stat-card-label">Aktuelle Patienten</div>
                </div>
                <div class="stat-card">
                    <div class="stat-card-header">
                        <div class="stat-card-icon green">
                            <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/></svg>
                        </div>
                    </div>
                    <div class="stat-card-value">${stats.today_admissions}</div>
                    <div class="stat-card-label">Aufnahmen heute</div>
                </div>
                <div class="stat-card">
                    <div class="stat-card-header">
                        <div class="stat-card-icon orange">
                            <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/></svg>
                        </div>
                    </div>
                    <div class="stat-card-value">${stats.open_invoices}</div>
                    <div class="stat-card-label">Offene Rechnungen</div>
                </div>
                <div class="stat-card">
                    <div class="stat-card-header">
                        <div class="stat-card-icon red">
                            <svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/></svg>
                        </div>
                    </div>
                    <div class="stat-card-value">${formatCurrency(stats.monthly_revenue)}</div>
                    <div class="stat-card-label">Umsatz diesen Monat</div>
                </div>
            `;
            document.getElementById('statsGrid').innerHTML = html;
        }

        function renderRecentPatients(patients) {
            if (patients.length === 0) {
                document.getElementById('recentPatientsTable').innerHTML = '<p style="text-align: center; color: var(--gray-500);">Keine Patienten vorhanden</p>';
                return;
            }

            const html = `
                <table>
                    <thead>
                        <tr>
                            <th>Patient</th>
                            <th>Zimmer</th>
                            <th>Arzt</th>
                            <th>Aufnahmedatum</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${patients.map(p => `
                            <tr>
                                <td><strong>${e(p.first_name)} ${e(p.last_name)}</strong></td>
                                <td>${e(p.room_number)}</td>
                                <td>${e(p.doctor)}</td>
                                <td>${formatDate(p.admission_date)}</td>
                                <td><span class="badge ${p.status === 'aufgenommen' ? 'badge-success' : 'badge-warning'}">${p.status === 'aufgenommen' ? 'Aufgenommen' : 'Entlassen'}</span></td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>
            `;
            document.getElementById('recentPatientsTable').innerHTML = html;
        }

        function renderRecentInvoices(invoices) {
            if (invoices.length === 0) {
                document.getElementById('recentInvoicesTable').innerHTML = '<p style="text-align: center; color: var(--gray-500);">Keine Rechnungen vorhanden</p>';
                return;
            }

            const html = `
                <table>
                    <thead>
                        <tr>
                            <th>Rechnungsnr.</th>
                            <th>Patient</th>
                            <th>Datum</th>
                            <th>Betrag</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${invoices.map(i => `
                            <tr>
                                <td><strong>${e(i.invoice_number)}</strong></td>
                                <td>${e(i.patient_name)}</td>
                                <td>${formatDate(i.invoice_date)}</td>
                                <td>${formatCurrency(i.total_amount)}</td>
                                <td><span class="badge badge-${i.status === 'bezahlt' ? 'success' : i.status === 'ueberfaellig' ? 'danger' : 'warning'}">${i.status}</span></td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>
            `;
            document.getElementById('recentInvoicesTable').innerHTML = html;
        }

        function formatCurrency(amount) {
            return new Intl.NumberFormat('de-DE', { style: 'currency', currency: 'EUR' }).format(amount);
        }

        function formatDate(dateString) {
            if (!dateString) return '-';
            return new Date(dateString).toLocaleDateString('de-DE');
        }

        function e(string) {
            if (!string) return '';
            return string.replace(/[&<>"']/g, function(m) {
                return {'&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#39;'}[m];
            });
        }

        function refreshData() {
            loadDashboardData();
        }

        function showError(message) {
            document.getElementById('statsGrid').innerHTML = `<p style="color: var(--danger); text-align: center;">${message}</p>`;
        }
    </script>
</body>
</html>